globals [
    whites        ;; number of white turtles
    blacks        ;; number of black turtles
    changed?      ;; has any turtle changed its colour?
    ]

turtles-own [
    strength      ;; of influence, set at initialisation to a random value
    ]

to setup
    clear-all
    ask patches [
        set pcolor green + 3    ;; make the background light green
        ;; make each patch sprout one turtle and set the turtle's colour etc.
        sprout 1 [
            ifelse (random 100 < initial-white)
                [set color white]
                [set color black]
            set strength random 100
            set heading 0
            set shape "person"
        ]
    ]
    countTurtles
end

to go
    set changed? false            ;; assume that no turtle will change its colour
    ask turtles [ beInfluenced ]
    countTurtles
    if (not changed?) [ stop ]    ;; stop if no turtle has changed its colour
end

to beInfluenced                    ;; turtle procedure
    ;; find the impact of supporters and opposers and change my colour
    ;; if the opposers' impact is greater than the supporters'

    let impactFromWhite 0
  let impactFromBlack 0

    set impactFromWhite  sum [impact myself] of turtles with [color = white]
    set impactFromBlack  sum [impact myself] of turtles with [color = black]

    ;; if the impact from whites is greater than that from blacks, change the turtle's colour to
    ;; white, and vice versa

    if color = black and (impactFromWhite > impactFromBlack)
        [set changed? true set color white]
    if color = white and (impactFromBlack > impactFromWhite)
        [set changed? true set color black]
end

to-report impact [otherTurtle]    ;; turtle procedure
    ;; report the impact of otherTurtle on myself
    let d 0

    set d distance otherTurtle
    if (d = 0) [set d self-distance]
        ;; (exp ((ln d) * exponent)) calculates d to the power 'exponent'
    report square (strength / (exp ((ln d) * exponent)))
end

to-report square [x]
    report x * x
end

to countTurtles
    set whites count turtles with [color = white]
    set blacks count turtles with [color = black]
end
@#$#@#$#@
GRAPHICS-WINDOW
378
10
768
421
9
9
20.0
1
10
1
1
1
0
1
1
1
-9
9
-9
9

SLIDER
76
134
248
167
initial-white
initial-white
0
100
50
1
1
%

BUTTON
73
71
154
104
setup
setup
NIL
1
T
OBSERVER
T
NIL
NIL

BUTTON
222
72
303
105
go
go
T
1
T
OBSERVER
T
NIL
NIL

TEXTBOX
75
116
253
134
Percent of white turtles at start

SLIDER
75
204
247
237
exponent
exponent
0
10
2
1
1
NIL

MONITOR
478
426
535
475
Whites
whites
0
1

MONITOR
603
425
660
474
Blacks
blacks
0
1

TEXTBOX
75
184
346
202
Exponent of the inverse power law for distance

TEXTBOX
36
29
345
56
SitSim: a re-implementation of a simulation to illustrate the Latane and Nowak theory of Dynamic Social Impact

SLIDER
75
268
247
301
self-distance
self-distance
0.1
2
0.8
0.1
1
NIL

TEXTBOX
76
250
290
268
Relative importance of self influence

@#$#@#$#@
WHAT IS IT?
-----------
This program models one theory of how people's attitudes are influenced by other people.  In the early 1980s, Bibb Latan? proposed a theory of 'social impact'.  The theory states that "the impact of other people on a given individual is a multiplicative function of the 'strength' of members of the group (how credible or persuasive they are), their immediacy (a decreasing function of their social distance from the individual) and their number." (Latan? 1996:65).

The theory has been extensively tested in a wide range of settings and has received a lot of support.  However, to understand its implications, we need to show not only the effect of other people on one individual, but also the effect of that individual on other people. This reciprocal influencing, where everyone influences everyone else, is more complicated to analyse, but can be shown simply in a simulation.

In this model, based on the original simulation called SITSIM (Nowak and Latan? 1994), turtles are distributed at random, each with a fixed, randomly assigned 'strength' (their persuasiveness).  Each turtle has one of two opinions, either 'white' or 'black' (e.g. they prefer white or red meat; they are for or against abortion; they vote on the left or the right).  Each turtle is then influenced by all the other turtles, changing its opinion if the sum of the impacts of the turtles opposing its current opinion is greater than the sum of the impacts of the turtles supporting its current opinion.

The impact of one turtle on another is calculated using an inverse power law, that is, the influence of one turtle on another is inversely proportional to their separation.  So a nearby turtle is more influential than one far away.  The influence also depends on the other turtle's persuasiveness, its 'strength'. In real life, the measure of separation may be physical distance, or it may be social distance (e.g. difference in status).

If a turtle changes its opinion, this will change the overall impacts experienced by all the other turtles. It might tip the balance so that another turtle changes its opinion and so on.  Eventually, however, the system will settle down to an equilibrium.


HOW TO USE IT
-------------
Press 'setup' to give each turtle an initial strength and an initial opinion.  Then press 'go' to get the turtles to influence each other.

Set INITIAL-WHITE to vary the percentage of turtles starting with a white rather than a black opinion.

Set EXPONENT to change the exponent of the inverse power law.  A setting of 2 gives an inverse square law: a turtle's influence on another turtle is proportional to the square root of the distance between them.  A setting of zero means that all distances are treated as equal (any number to the power of zero is equal to 1).  A high setting means that only the nearest neighbours have any significant influence.

Set SELF-DISTANCE to change the effect of the turtle's own opinion on itself.  If the value is low, the effect of other turtles' opinions is small, and the turtle will only change its opinion if there is a lot of opposing impact.  If the value is high, self-influence becomes negligible.  The original model used a distance to self of 0.84, meaning that the self is somewhat more influential than any other single turtle (the closest any other turtle can be is 1).

THINGS TO NOTICE
----------------
With the default settings (INITIAL-WHITE 50%, EXPONENT 2, SELF-DISTANCE 0.8), the turtles form into clumps  sharing the same opinion.  Although the initial equal division of opinion changes to a polarised one in which either black or white are in the majority, the minority is never overwhelmed by the majority. There are always some minority opinions remaining.

Run the model a few times and check that the minority always survives.  How big is the minority?

Usually, there are one or two isolated turtles of one colour, surrounded by turtles of the other colour.  How do they manage to resist the opposing opinions of their neighbours?  (Hint:  select an isolate and use a Turtle Monitor to examine their attributes).

THINGS TO TRY
-------------
Use the INITIAL-WHITE slider to reduce the number of turtles that hold the black opinion at the start.  Notice that when the starting percentage of blacks is very small, there is a high chance that the black opinion will disappear entirely by the end of the run.  How small can the initial percentage of black turtles be, yet still survive most of the time?

Return to an INITIAL-WHITE setting of 50% and investigate the effect of changing the inverse power law EXPONENT.  What is the effect of setting the slider to 0?  And to 10?  Nowak found that as a general rule, the exponent has to be no smaller than the dimensionality of the social space (which is 2 in this model, because the turtles are on a two-dimensional grid) in order to obtain clustering.

Explore the effect of varying the SELF-DISTANCE.  Setting it to a high value means that the turtle's own opinion counts for little in comparison with the other turtles' opinions.

EXTENDING THE MODEL
-------------------
Here are some suggestions for ways to extend the model.

- Introduce some 'noise'.  In the real world, social influences are never communicated perfectly.  What happens if the impact of one turtle on another is affected by a small random 'error' (which could be either positive or negative)?

- In the model, initial strengths are assigned at random from a uniform distribution and stay constant throughout the run.  Investigate the effect of:
  * giving every turtle the same strength
  * giving a few turtles a high strength and rest a low strength (this models a setting in which there are a few influential leaders and the rest are less influential followers)

- Harder: Introduce a second attitude, which turtles hold independently of the first, but which follows the same rules of social impact.  For example, if first attitude is about whether turtles have a white or black opinion, the second attitude might be about whether they favour blue or green.  Compare the opinion clusters for the two attitudes. At the end of a run, do those turtles with a given opinion for one attitude tend to share the same opinion on the second attitude (e.g. do white turtles tend to be also those with red opinions)?

See Nowak and Latan? (1994) for many other suggestions for parameters that can be varied and their effects.


RELATED MODELS
--------------
Cellular automata models, but notice that while the cells in CA models are only affected by their closest neighbours, in this model, every turtle is affected to some degree by every other turtle.


CREDITS AND REFERENCES
----------------------
The original social impact theory is described in Latan?, B. (1981) The psychology of social impact.  American Psychologist, 36: 343-356

The SITSIM model is described in Nowak, A. and Latan?, B (1994) Simulating the emergence of social order from individual behaviour, in N. Gilbert and J. Doran, Simulating Societies, London: UCL Press.

The following parameters are used in this model (these are the SitSim default settings):
   Accumulative impact formula; No borders; Full window;
   Monte Carlo update; Unit persuasiveness and supportiveness;
   Uniform distribution of strength; Same coupling; Same reassignment;
   No bias; No pre-clustering;
   Infinite non-linearity coefficient; One attitude; Zero temperature

See also N. Gilbert and K.G. Troitzsch (1991) Simulation for the social scientist, Open University Press, chapter 7.

AUTHOR
------
Nigel Gilbert August 22, 1996 (version in XLisp); July 25, 2002 (version in NetLogo); August 8, 2003 (revised)
contact: n.gilbert@soc.surrey.ac.uk  http://www.soc.surrey.ac.uk/staff/nigel_gilbert.html
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

ant
true
0
Polygon -7566196 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7566196 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7566196 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7566196 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7566196 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7566196 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7566196 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7566196 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7566196 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee
true
0
Polygon -256 true false 152 149 77 163 67 195 67 211 74 234 85 252 100 264 116 276 134 286 151 300 167 285 182 278 206 260 220 242 226 218 226 195 222 166
Polygon -16777216 true false 150 149 128 151 114 151 98 145 80 122 80 103 81 83 95 67 117 58 141 54 151 53 177 55 195 66 207 82 211 94 211 116 204 139 189 149 171 152
Polygon -7566196 true true 151 54 119 59 96 60 81 50 78 39 87 25 103 18 115 23 121 13 150 1 180 14 189 23 197 17 210 19 222 30 222 44 212 57 192 58
Polygon -16777216 true false 70 185 74 171 223 172 224 186
Polygon -16777216 true false 67 211 71 226 224 226 225 211 67 211
Polygon -16777216 true false 91 257 106 269 195 269 211 255
Line -1 false 144 100 70 87
Line -1 false 70 87 45 87
Line -1 false 45 86 26 97
Line -1 false 26 96 22 115
Line -1 false 22 115 25 130
Line -1 false 26 131 37 141
Line -1 false 37 141 55 144
Line -1 false 55 143 143 101
Line -1 false 141 100 227 138
Line -1 false 227 138 241 137
Line -1 false 241 137 249 129
Line -1 false 249 129 254 110
Line -1 false 253 108 248 97
Line -1 false 249 95 235 82
Line -1 false 235 82 144 100

bird1
false
0
Polygon -7566196 true true 2 6 2 39 270 298 297 298 299 271 187 160 279 75 276 22 100 67 31 0

bird2
false
0
Polygon -7566196 true true 2 4 33 4 298 270 298 298 272 298 155 184 117 289 61 295 61 105 0 43

boat1
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 158 33 230 157 182 150 169 151 157 156
Polygon -7566196 true true 149 55 88 143 103 139 111 136 117 139 126 145 130 147 139 147 146 146 149 55

boat2
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 157 54 175 79 174 96 185 102 178 112 194 124 196 131 190 139 192 146 211 151 216 154 157 154
Polygon -7566196 true true 150 74 146 91 139 99 143 114 141 123 137 126 131 129 132 139 142 136 126 142 119 147 148 147

boat3
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7566196 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

box
true
0
Polygon -7566196 true true 45 255 255 255 255 45 45 45

butterfly1
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

circle
false
0
Circle -7566196 true true 35 35 230

person
false
0
Circle -7566196 true true 155 20 63
Rectangle -7566196 true true 158 79 217 164
Polygon -7566196 true true 158 81 110 129 131 143 158 109 165 110
Polygon -7566196 true true 216 83 267 123 248 143 215 107
Polygon -7566196 true true 167 163 145 234 183 234 183 163
Polygon -7566196 true true 195 163 195 233 227 233 206 159

spacecraft
true
0
Polygon -7566196 true true 150 0 180 135 255 255 225 240 150 180 75 240 45 255 120 135

thin-arrow
true
0
Polygon -7566196 true true 150 0 0 150 120 150 120 293 180 293 180 150 300 150

truck-down
false
0
Polygon -7566196 true true 225 30 225 270 120 270 105 210 60 180 45 30 105 60 105 30
Polygon -8716033 true false 195 75 195 120 240 120 240 75
Polygon -8716033 true false 195 225 195 180 240 180 240 225

truck-left
false
0
Polygon -7566196 true true 120 135 225 135 225 210 75 210 75 165 105 165
Polygon -8716033 true false 90 210 105 225 120 210
Polygon -8716033 true false 180 210 195 225 210 210

truck-right
false
0
Polygon -7566196 true true 180 135 75 135 75 210 225 210 225 165 195 165
Polygon -8716033 true false 210 210 195 225 180 210
Polygon -8716033 true false 120 210 105 225 90 210

turtle
true
0
Polygon -7566196 true true 138 75 162 75 165 105 225 105 225 142 195 135 195 187 225 195 225 225 195 217 195 202 105 202 105 217 75 225 75 195 105 187 105 135 75 142 75 105 135 105

wolf-left
false
3
Polygon -6524078 true true 117 97 91 74 66 74 60 85 36 85 38 92 44 97 62 97 81 117 84 134 92 147 109 152 136 144 174 144 174 103 143 103 134 97
Polygon -6524078 true true 87 80 79 55 76 79
Polygon -6524078 true true 81 75 70 58 73 82
Polygon -6524078 true true 99 131 76 152 76 163 96 182 104 182 109 173 102 167 99 173 87 159 104 140
Polygon -6524078 true true 107 138 107 186 98 190 99 196 112 196 115 190
Polygon -6524078 true true 116 140 114 189 105 137
Rectangle -6524078 true true 109 150 114 192
Rectangle -6524078 true true 111 143 116 191
Polygon -6524078 true true 168 106 184 98 205 98 218 115 218 137 186 164 196 176 195 194 178 195 178 183 188 183 169 164 173 144
Polygon -6524078 true true 207 140 200 163 206 175 207 192 193 189 192 177 198 176 185 150
Polygon -6524078 true true 214 134 203 168 192 148
Polygon -6524078 true true 204 151 203 176 193 148
Polygon -6524078 true true 207 103 221 98 236 101 243 115 243 128 256 142 239 143 233 133 225 115 214 114

wolf-right
false
3
Polygon -6524078 true true 170 127 200 93 231 93 237 103 262 103 261 113 253 119 231 119 215 143 213 160 208 173 189 187 169 190 154 190 126 180 106 171 72 171 73 126 122 126 144 123 159 123
Polygon -6524078 true true 201 99 214 69 215 99
Polygon -6524078 true true 207 98 223 71 220 101
Polygon -6524078 true true 184 172 189 234 203 238 203 246 187 247 180 239 171 180
Polygon -6524078 true true 197 174 204 220 218 224 219 234 201 232 195 225 179 179
Polygon -6524078 true true 78 167 95 187 95 208 79 220 92 234 98 235 100 249 81 246 76 241 61 212 65 195 52 170 45 150 44 128 55 121 69 121 81 135
Polygon -6524078 true true 48 143 58 141
Polygon -6524078 true true 46 136 68 137
Polygon -6524078 true true 45 129 35 142 37 159 53 192 47 210 62 238 80 237
Line -16777216 false 74 237 59 213
Line -16777216 false 59 213 59 212
Line -16777216 false 58 211 67 192
Polygon -6524078 true true 38 138 66 149
Polygon -6524078 true true 46 128 33 120 21 118 11 123 3 138 5 160 13 178 9 192 0 199 20 196 25 179 24 161 25 148 45 140
Polygon -6524078 true true 67 122 96 126 63 144

@#$#@#$#@
NetLogo 4.0pre3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
