breed [persons person]
breed [viruses virus]
breed [plants plant]


persons-own [
  home-x
  home-y
  incubation-period          ;each person is assigned an incubation period at the beginning. This variable is used to determine how soon the people change from exposed to infectious state.
  incubation-period-value    ;this variable stores the original incubation period and is used to check the average incubation period.
  infectious-period          ;This variable is used to track people's disease periods
  contacts                   ;number of susceptible contacts in defined infection radius
  infected-by-me             ;total number of people I infect (numerical results)
  infect-any?                ;whether I have infected any one (true/false)
]

viruses-own [
  viability-period   ;viral viability in treatment plant
  in-plant?          ;check if the virus has been in the plant  (true/false)
]

Patches-own [
  p-type      ;define patch type  0 = town/community, 1 = pipe, 2 = plant, 3 = wild area, 4 = creek, 5 = testing location
]

Globals [
  main-pipe-xcor        ;the most right vertical pipe
  main-pipe-ycor        ;the lowest horizental pipe
  max-daily-cases
  infection-number      ;total of people are infected
  infector-number       ;total of people who infected at least one other person
  virus-yield           ;record the number of viruses that are produced
  start-record?         ;trigger data recording
  prevalence-list       ;This list stores the cumulative infected (include both E and I states) once local survaillence is triggered.
  RNA-copies            ;Coronvirous RNA copies
  individual-RNA-contribution    ;Copies of RNA increase by each person
  R0-list               ;create a list of ongoing R0
  death-toll

]


to-report R0
  if any? persons with [infect-any? = true]
    [report infection-number / infector-number]   ;This is the effective R0. A cross check is the total infected (E and I and R) dvided by total people with [infect-any? = true]
end

to-report c
  report mean [contacts] of persons with [color = 26]   ;only E people may infect others so only count the susceptibles of E
end

to-report infectious-period-of-exposed
 report mean [incubation-period-value] of persons with [color = 26]   ;avergage incubation periods of exposed people .
end

to-report R-t
  report (0.01 * transmission-rate) * c *  infectious-period-of-exposed   ;only E may infect others so use incubation period rather than disease period to calculate Rt
end


to-report sewer-patches
  report patches with [p-type = 1]
end

to-report susceptible-people
  report persons with [color = 55]
end

to-report exposed-people
  report persons with [color = 26]
end

to-report infectious-people           ;these are symptomatic people. they are quarantined or isolated so they cannot infect others.
  report persons with [color = red]
end

to-report recovered-people
  report persons with [color = sky]
end

to-report local-surveillance-infection     ;report the first prevalence in the prevalence list
  if ticks > 0 and not empty? prevalence-list
   [report item 0 prevalence-list]
end

to-report central-surveillance-infection     ;report the last prevalence in the prevalence list
  if ticks > 0 and length prevalence-list = Central-testing-duration
   [report item (central-testing-duration - 1) prevalence-list]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;   Setup procedures
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to setup
  clear-all
  set-default-shape viruses "c-virus-0"
  set-default-shape persons "person-0"

  set-patches
  setup-people
  add-an-carrier
  find-contacts

  set start-record? false
  set prevalence-list []
  set R0-list []
  set death-toll 0

  reset-ticks
end

to set-patches
  ask patches [set pcolor white set p-type 0 ]  ;set background

  ask patches with [pycor <= 40] [set pcolor 9 set p-type 3] ;set wild field

  ask patches with [pycor <= min-pxcor + 4] [set pcolor 105 set p-type 4]  ;set a creek

  ;set horizontal pipe
  ask patches
   [if remainder (pycor - 2) 25 = 0 and pycor > 40 and pycor <= max-pycor and pxcor >= 2 and pxcor <= max-pxcor - 2
   [set pcolor 89 set p-type 1]]

  ;set vertical pipes
  ask patches
  [if remainder (pxcor - 2) 35 = 0 and pxcor >= 10 and pxcor <= max-pxcor and pycor <= max-pycor - 2 and pycor >= 45
   [set pcolor 89 set p-type 1]]

  ;connect to plant
  if any? patches with [p-type = 1]
  [let y min [pycor] of patches with [p-type = 1]
   let lowest-patch patches with [p-type = 1 and pycor = y]
   ;let connecting-patch lowest-patch with-max [pxcor]
   let x max [pxcor] of lowest-patch

   ask patches with [pxcor = x and pycor = y] [set pcolor 88]   ;label the correct connecting point

   ask patches with [pxcor = x and pycor < y and pycor >= min-pycor + 20 ] [set pcolor 89 set p-type 1]   ;extend the pipline


   ;set the plant
   create-plants 1
   [set shape "w-plant"
    set size 14
    set color 89
    setxy x 11.5]

   set main-pipe-xcor x    ;record the main-pipe xcor
   set main-pipe-ycor y    ;record the lowest pipe's ycor

    ask patches with [pxcor = main-pipe-xcor and pycor = 20] [set pcolor red]

  ]

end

to setup-people
  ask n-of population patches with [p-type = 0]
  [sprout-persons 1
    [set size 2.75
     set color 55
     set shape "person-0"
     set home-x xcor
     set home-y ycor
     set incubation-period precision (random-normal 6 2) 0  ;set an incubation period based on normal distribution a mean of 6 and a SD of 2
     set incubation-period-value incubation-period   ;set an incubation period value for tracking the change in incubation period
     set infectious-period 0
     set infected-by-me 0
     set infect-any? false
     ]
  ]

  ;set individual RNA shed contribution based on population size. Equation from Data from Marc Johnson in MO 9The 10^9 copies of RNA per infected person gets diluted into the bulk wastewater.)
  set individual-RNA-contribution (1000 / (1.1355 * population - 900))  ;900 is used to adjust the function for running a smaller population

end

to add-an-carrier
  ask one-of persons
  [set color 26 ]

end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;   Go procedures
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to go

  if ticks > 0 and (not any? exposed-people and not any? infectious-people) [stop]

  set RNA-copies 0       ;reset the number of RNA-copies at the testing site each tick
  move-viruses    ;put move-virus before produce-viruses make the newly produced viruses arrive the plant the next day or later.
  move-people

 (ifelse
    Issue-lockdown? = "none"    ;This is for comparing local and certral surveillance
      [SEIR]

   Issue-lockdown? = "local"    ;This is for gathering LOCAL alert prevalence
    [if length prevalence-list = 1
      [set Transmission-rate 1
        set %-travelers 5]

        SEIR]

    Issue-lockdown? = "central"    ;This is for gathering CENTRAL alert prevalence
    [if length prevalence-list = Central-testing-duration
      [set Transmission-rate 1
        set %-travelers 5]

       SEIR])

  find-contacts
  find-max-daily-cases
  calculate-average-R0

  surveillance
  disinfected
  back-home

  plot-infection-over-R0

 tick

end

to SEIR
  incubation    ;E->I     ;put incubation before infect to update incubation period in the next day
  infect        ;S->E
  produce-viruses
  recover       ;I->E
end

to move-people
 ask up-to-n-of ((count persons with [color != red]) * %-travelers * 0.01) persons with [color != red]   ;let up to 10% of the people who are not symptomatics to randomly move
  [setxy random-xcor (41 + random (max-pycor - 41))]
end

to move-viruses              ;viruses arrive plant in a day
  ask viruses with [xcor = main-pipe-xcor and ycor = 20]   ;move the virues at the testing site down one patch to empty the testing site
    [setxy main-pipe-xcor 19]

  ask viruses with [xcor != main-pipe-xcor or ycor > 20]   ;move the newly produced viruses to the testing site
    [setxy main-pipe-xcor 20 set in-plant? true]

end


to produce-viruses

    ask exposed-people                ;ask exposed-infected people
    [let sewer min-one-of sewer-patches [distance myself]  ;find the closest sewer
     if random 100 < 43
     [ hatch-viruses 1                               ;produce a new virus at a chance of 43%
       [set size 0.75                                 ;set the size of a virus
        set color 35
        setxy [pxcor] of sewer  [pycor] of sewer
        set viability-period 1
        set virus-yield virus-yield + 1
        set in-plant? false
       ]
     ]]

end

to infect    ; S->E

  ask exposed-people; with [contacts > 0]
      [if any? susceptible-people in-radius infection-radius           ;if any susceptible people nearby
       [repeat count susceptible-people in-radius infection-radius     ;repeat the times of susceptible contacts to contact all these contacts
          [if random 1000 < (Transmission-Rate  * 10)                      ;at the chance of transmission rate
             [let infected-person one-of susceptible-people in-radius infection-radius      ;pick one of the susceptible people in the infection radius
              ask infected-person                               ;infect him/her
                 [set color 26]                                 ;change color. No need to set the infection-period as it is updated in recover procedure

              set infected-by-me infected-by-me + 1             ;update how many people I have infected
              if infect-any? = false                            ;if it is new infector
                [set infect-any? true                           ;update infection record
                 set infector-number infector-number + 1]       ;update total number of of infectors


              set infection-number infection-number + 1]]]]     ;update total number of infection

end

to incubation     ;E->I

  ask exposed-people
     [ifelse incubation-period > 0
        [set incubation-period (incubation-period - 1)]
        [set color red]]

end


to recover   ;I->R
  ask infectious-people
  [ifelse infectious-period >= 14    ; set disease period as 2 weeks. This period does not affect R0 as it is assumed that infected people are well isolated and do not transmit the disease.
    [ifelse random 100 < 3           ; mortality equals 3%
      [set death-toll death-toll + 1 die]
      [set color sky set infectious-period 0]]
    [set infectious-period infectious-period + 1]
  ]

end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;   supporting procedures
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;



to find-max-daily-cases
  if count infectious-people > max-daily-cases        ;Count the infectious.If it is greater than the current record of max daily cases
  [set max-daily-cases count infectious-people]       ;update the max daily case
end

to disinfected

  ask viruses with [xcor = main-pipe-xcor and ycor = 19]    ; ask viruses in the treatment plant to die after 1 days ro save compute power.
    [ifelse viability-period > 0
      [set viability-period viability-period - 1]
      [die]]

end

to find-contacts
  ask persons
  [set contacts count other persons with [color = 55] in-radius infection-radius]
end

to surveillance
  if (count viruses-on patch main-pipe-xcor 20) * individual-RNA-contribution  >= Surveillance-threshold
    [set start-record? true]

 if start-record? = true
    [if length prevalence-list < Central-testing-duration
        [ set prevalence-list lput (precision (infection-number * 100 / Population) 2) prevalence-list]]

end

to plot-infection-over-R0   ;;; this creates creates the scotter plot graph

    set-current-plot "Prevalence under Local vs. Central Surveillance"
    clear-plot
    plot-pen-down
  set-current-plot-pen "Day 1(Local)"
    if any? persons with [infect-any? = true] and length prevalence-list >= 1 [plotxy 1 item 0 prevalence-list]
  set-current-plot-pen "Day 2"
    if any? persons with [infect-any? = true] and length prevalence-list >= 2 [plotxy 2 item 1 prevalence-list]
  set-current-plot-pen "Day 3"
    if any? persons with [infect-any? = true] and length prevalence-list >= 3 [plotxy 3 item 2 prevalence-list]
  set-current-plot-pen "Day 4(Central)"
    if any? persons with [infect-any? = true] and length prevalence-list >= 4 [plotxy 4 item 3 prevalence-list]
  set-current-plot-pen "Day 5(Central)"
    if any? persons with [infect-any? = true] and length prevalence-list >= 5 [plotxy 5 item 4 prevalence-list]
  set-current-plot-pen "Day 6(Central)"
    if any? persons with [infect-any? = true] and length prevalence-list >= 6 [plotxy 6 item 5 prevalence-list]
  set-current-plot-pen "Day 7(Central)"
    if any? persons with [infect-any? = true] and length prevalence-list >= 7 [plotxy 7 item 6 prevalence-list]
end

to calculate-average-R0
 if any? exposed-people with [infect-any? = true ] [set R0-list lput (precision R0 2) R0-list]
end

to back-home
  ask persons with [xcor != home-x or ycor != home-y]
  [setxy home-x home-y]
End
@#$#@#$#@
GRAPHICS-WINDOW
50
159
559
696
-1
-1
3.0
1
10
1
1
1
0
0
0
1
0
166
0
175
0
0
1
Days
5.0

BUTTON
883
26
985
63
Start/Reset
setup\n\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
991
26
1083
63
Run/Pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
991
70
1083
108
Run a day
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
51
33
241
66
Population
Population
1000
5000
1000.0
50
1
People
HORIZONTAL

PLOT
568
195
1007
440
Changes in the Population over Time
Days
% of population
0.0
10.0
0.0
100.0
true
true
"" ""
PENS
"Total Cumulative infectious" 1.0 0 -13840069 true "" "plot infection-number * 100 / population  "
"Daily Cumulative Exposed" 1.0 0 -955883 true "" "plot count persons with [color = 26] * 100 / population  "
"Daily Cumulative Infectious" 1.0 0 -2674135 true "" "plot count persons with [color = red] * 100 / population  "
"Daily New Infectious Cases" 1.0 0 -13791810 true "" "plot count infectious-people with [infectious-period = 1] * 100 / population"

MONITOR
1009
203
1091
240
Susceptible (%)
count persons with [color = 64] * 100 / population
2
1
9

MONITOR
1009
317
1091
354
Recovered (%)
count persons with [color = sky] * 100 / population
2
1
9

MONITOR
1009
279
1091
316
Infectious (%)
count persons with [color = red] * 100 / population
2
1
9

MONITOR
1009
357
1091
394
Death (%)
death-toll * 100 / population
2
1
9

TEXTBOX
53
702
121
720
SEIR colors:
10
125.0
1

TEXTBOX
110
704
204
722
|Green:susceptible
9
64.0
1

TEXTBOX
282
702
352
720
|Red:infectious
9
15.0
1

TEXTBOX
353
702
429
720
|Blue:recovered
9
94.0
1

MONITOR
568
696
819
733
RNA detected at the treatment plant (Copies/mL)
(count viruses-on patch main-pipe-xcor 20) * individual-RNA-contribution
1
1
9

SLIDER
51
68
241
101
Infection-radius
Infection-radius
1
15
7.0
0.1
1
NIL
HORIZONTAL

MONITOR
568
150
664
187
Re (Effective R0)
R0
2
1
9

SLIDER
434
33
715
66
Surveillance-threshold
Surveillance-threshold
50
100
50.0
1
1
RNA Copies/mL
HORIZONTAL

MONITOR
1009
395
1091
432
Prevalence (%)
(infection-number + 1) * 100 / Population
2
1
9

SLIDER
434
67
715
100
Central-testing-duration
Central-testing-duration
0
7
7.0
1
1
Days
HORIZONTAL

TEXTBOX
203
10
291
28
R0 parameters:
11
125.0
1

TEXTBOX
517
13
667
31
Surveillance parameters:
11
125.0
1

SLIDER
244
33
425
66
Transmission-rate
Transmission-rate
0
20
10.0
0.1
1
%
HORIZONTAL

PLOT
568
444
1007
672
Prevalence under Local vs. Central Surveillance
Day
Prevalence (%)
1.0
8.0
0.0
1.0
true
true
"" ""
PENS
"Day 1(Local)" 1.0 1 -13345367 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 1 [plotxy 1 item 0 prevalence-list]"
"Day 2" 1.0 1 -13791810 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 2 [plotxy 2 item 1 prevalence-list]"
"Day 3" 1.0 1 -11221820 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 3 [plotxy 3 item 2 prevalence-list]"
"Day 4(Central)" 1.0 1 -11085214 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 4 [plotxy 4 item 3 prevalence-list]"
"Day 5(Central)" 1.0 1 -4399183 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 5 [plotxy 5 item 4 prevalence-list]"
"Day 6(Central)" 1.0 1 -1184463 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 6 [plotxy 6 item 5 prevalence-list]"
"Day 7(Central)" 1.0 1 -817084 true "" "if any? persons with [infect-any? = true] and length prevalence-list >= 7 [plotxy 7 item 6 prevalence-list]"

MONITOR
665
150
785
187
Rt (time-dependent R0)
R-t
2
1
9

TEXTBOX
567
680
717
698
Pathogen Counts:
11
125.0
1

TEXTBOX
569
131
719
149
Reproduction Number:
11
125.0
1

SLIDER
244
68
425
101
%-travelers
%-travelers
0
25
10.0
1
1
%
HORIZONTAL

TEXTBOX
200
704
281
722
|Orange:Exposed
9
25.0
1

MONITOR
823
696
1084
733
Individual RNA copy Contribution (Copies/mL) 
individual-RNA-contribution
1
1
9

MONITOR
1010
479
1092
516
Day 1 (Local)
item 0 prevalence-list
17
1
9

MONITOR
1010
517
1092
554
Day 4 (Central)
item 3 prevalence-list
17
1
9

MONITOR
1010
556
1092
593
Day 5 (Central)
item 4 prevalence-list
17
1
9

MONITOR
1010
595
1092
632
Day 6 (Central)
item 5 prevalence-list
17
1
9

MONITOR
1010
634
1092
671
Day 7 (Central)
item 6 prevalence-list
17
1
9

MONITOR
1009
242
1091
279
Exposed (%)
count persons with [color = 26] * 100 / population
2
1
9

CHOOSER
735
44
865
89
Issue-lockdown?
Issue-lockdown?
"none" "local" "central"
0

BUTTON
884
70
985
108
Default Settings
set population 1000\nset infection-radius 7\nset Transmission-rate 10\nset %-travelers 10\nset Surveillance-threshold 50\nset Central-testing-duration 7
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

TEXTBOX
1011
464
1098
482
Prevalence (%):
11
125.0
1

MONITOR
786
150
933
187
Avg. susceptible contacts
mean [contacts] of persons
2
1
9

@#$#@#$#@
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

bush
false
0
Polygon -6459832 true false 135 197 129 300 191 299 183 194
Circle -7500403 true true 103 113 132
Circle -7500403 true true 123 4 94
Circle -7500403 true true 71 35 108
Circle -7500403 true true 138 52 127
Circle -7500403 true true 189 101 120
Circle -7500403 true true -1 91 150

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

c-virus
true
0
Circle -7500403 true true 53 51 196
Circle -7500403 true true 203 234 16
Polygon -7500403 true true 43 208 82 179 92 196 46 215
Polygon -7500403 true true 257 208 218 179 208 196 254 215
Polygon -7500403 true true 43 92 82 121 92 104 46 85
Polygon -7500403 true true 257 92 218 121 208 104 254 85
Circle -7500403 true true 251 73 22
Polygon -7500403 true true 146 271 141 225 160 226 155 271
Polygon -7500403 true true 146 29 141 75 160 74 155 29
Polygon -7500403 true true 209 245 188 221 195 209 213 241
Polygon -7500403 true true 209 55 188 79 195 91 213 59
Circle -7500403 true true 203 50 16
Circle -7500403 true true 81 50 16
Polygon -7500403 true true 91 55 112 79 105 91 87 59
Polygon -7500403 true true 91 245 112 221 105 209 87 241
Circle -7500403 true true 81 234 16
Polygon -7500403 true true 260 147 230 142 229 157 260 154
Circle -7500403 true true 256 143 16
Circle -7500403 true true 28 143 16
Polygon -7500403 true true 40 147 70 142 71 157 40 154
Circle -7500403 true true 27 73 22
Circle -7500403 true true 27 205 22
Circle -7500403 true true 251 205 22
Circle -7500403 true true 139 270 22
Circle -7500403 true true 139 8 22
Polygon -6459832 true false 135 90 180 120 150 135 180 150 150 165 180 180 135 210 135 210 120 210 165 180 135 165 165 150 135 135 165 120 120 90

c-virus-0
true
0
Circle -7500403 true true 59 58 182
Polygon -6459832 true false 135 90 180 120 150 135 180 150 150 165 180 180 135 210 135 210 120 210 165 180 135 165 165 150 135 135 165 120 120 90
Polygon -7500403 true true 15 165 15 135 30 126 60 150 30 173
Polygon -7500403 true true 204 257 192 261 184 254 188 234 205 247
Polygon -7500403 true true 135 15 165 15 174 30 150 60 127 30
Polygon -7500403 true true 285 135 285 165 270 174 240 150 270 127
Polygon -7500403 true true 165 285 135 285 126 270 150 240 173 270
Polygon -7500403 true true 66 253 45 235 48 221 85 213 80 248
Polygon -7500403 true true 43 204 39 192 46 184 66 188 53 205
Polygon -7500403 true true 96 257 108 261 116 254 112 234 95 247
Polygon -7500403 true true 257 96 261 108 254 116 234 112 247 95
Polygon -7500403 true true 257 204 261 192 254 184 234 188 247 205
Polygon -7500403 true true 43 96 39 108 46 116 66 112 53 95
Polygon -7500403 true true 96 43 108 39 116 46 112 66 95 53
Polygon -7500403 true true 204 43 192 39 184 46 188 66 205 53
Polygon -7500403 true true 47 66 65 45 79 48 87 85 52 80
Polygon -7500403 true true 234 47 255 65 252 79 215 87 220 52
Polygon -7500403 true true 253 234 235 255 221 252 213 215 248 220

c-virus-1
true
0
Circle -7500403 true true 44 43 212
Polygon -6459832 true false 135 90 180 120 150 135 180 150 150 165 180 180 135 210 135 210 120 210 165 180 135 165 165 150 135 135 165 120 120 90
Polygon -7500403 true true 285 135 285 165 272 155 240 150 272 145
Polygon -7500403 true true 135 15 165 15 155 28 150 60 145 28
Polygon -7500403 true true 135 285 165 285 155 272 150 240 145 272
Polygon -7500403 true true 15 135 15 165 28 155 60 150 28 145
Polygon -7500403 true true 60 267 31 240 54 237 79 218 61 244
Polygon -7500403 true true 33 60 60 31 63 54 82 79 56 61
Polygon -7500403 true true 240 33 269 60 246 63 221 82 239 56
Polygon -7500403 true true 267 240 240 269 237 246 218 221 244 239
Polygon -7500403 true true 210 39 191 30 193 41 186 66 198 44
Polygon -7500403 true true 90 39 109 30 107 41 114 66 102 44
Polygon -7500403 true true 261 90 270 109 259 107 234 114 256 102
Polygon -7500403 true true 210 261 191 270 193 259 186 234 198 256
Polygon -7500403 true true 39 210 30 191 41 193 66 186 44 198
Polygon -7500403 true true 261 210 270 191 259 193 234 186 256 198
Polygon -7500403 true true 90 261 109 270 107 259 114 234 102 256
Polygon -7500403 true true 39 90 30 109 41 107 66 114 44 102

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7500403 true true 76 194 285 270
Rectangle -7500403 true true 36 95 59 231
Rectangle -16777216 true false 90 210 270 240
Line -7500403 true 90 195 90 255
Line -7500403 true 120 195 120 255
Line -7500403 true 150 195 150 240
Line -7500403 true 180 195 180 255
Line -7500403 true 210 210 210 240
Line -7500403 true 240 210 240 240
Line -7500403 true 90 225 270 225
Circle -1 true false 37 73 32
Circle -1 true false 55 38 54
Circle -1 true false 96 21 42
Circle -1 true false 105 40 32
Circle -1 true false 129 19 42
Rectangle -7500403 true true 14 228 78 270

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person-0
false
0
Circle -7500403 true true 75 0 150
Polygon -7500403 true true 151 166 178 165 203 163 225 164 243 167 262 175 279 189 292 208 299 230 300 249 298 275 293 290 260 295 205 299 151 299 95 298 38 295 5 287 2 270 0 248 4 226 14 202 29 183 42 172 57 165 76 163 99 163 124 164 151 166

person-1
false
0
Circle -7500403 true true 74 0 150
Polygon -7500403 true true 151 166 178 165 203 163 225 164 243 167 262 175 279 189 292 208 299 230 300 249 298 275 293 290 260 295 205 299 151 299 95 298 38 295 5 287 2 270 0 248 4 226 14 202 29 183 42 172 57 165 76 163 99 163 124 164 151 166
Polygon -1 true false 74 66 92 71 148 62 206 73 223 67 225 71 208 77 203 124 202 128 200 131 193 136 149 145 106 136 102 133 97 129 94 123 89 75 74 71

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

w-plant
false
3
Rectangle -7500403 true false 21 59 39 101
Circle -7500403 true false 196 93 90
Circle -7500403 true false 194 210 90
Circle -7500403 true false 89 210 90
Circle -11221820 true false 197 213 84
Polygon -7500403 true false 30 30 75 30 90 30
Circle -7500403 true false 91 93 90
Circle -11221820 true false 199 96 84
Circle -11221820 true false 94 96 84
Rectangle -7500403 true false 73 188 291 200
Rectangle -7500403 true false -16 90 74 294
Polygon -13345367 true false 15 -15 15 60 45 75 45 0 15 -15
Polygon -6459832 true true 45 0 270 0 270 75 45 75
Polygon -11221820 true false 15 -15 240 -15 270 0 45 0
Rectangle -11221820 true false 255 15 270 44
Rectangle -7500403 true false 210 15 240 74
Rectangle -11221820 true false 180 15 195 53
Rectangle -11221820 true false 150 15 165 45
Rectangle -11221820 true false 120 15 136 51
Rectangle -11221820 true false 90 15 105 45
Rectangle -11221820 true false 60 15 75 53
Circle -11221820 true false 92 213 84
Circle -6459832 true true 119 240 30
Polygon -7500403 true false 187 197 136 257 133 254 184 193
Circle -6459832 true true 119 120 30
Polygon -7500403 true false 133 137 184 197 187 194 136 133
Circle -6459832 true true 226 120 30
Polygon -7500403 true false 189 193 240 133 243 136 192 197
Circle -6459832 true true 224 240 30
Polygon -7500403 true false 188 197 239 257 242 254 191 193
Rectangle -13791810 true false -1 105 60 139
Rectangle -13791810 true false -1 154 60 188
Rectangle -13791810 true false -1 199 60 233
Rectangle -13791810 true false -1 244 60 278

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Test p6000" repetitions="40" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p6000-local" repetitions="90" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;local&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p6000-Central" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;central&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p10000-local" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;local&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p10000-Central" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;central&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p10000-local-central" repetitions="90" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;local&quot;"/>
      <value value="&quot;central&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p6000-local-central" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;local&quot;"/>
      <value value="&quot;central&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p6000-none" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p10000-none" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Test p6000-none-test table" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>R0</metric>
    <metric>mean R0-list</metric>
    <metric>item 0 prevalence-list</metric>
    <metric>item 3 prevalence-list</metric>
    <metric>item 4 prevalence-list</metric>
    <metric>item 5 prevalence-list</metric>
    <metric>item 6 prevalence-list</metric>
    <metric>infection-number * 100 / Population</metric>
    <enumeratedValueSet variable="Central-testing-duration">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Infection-radius">
      <value value="4.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-travelers">
      <value value="45"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Population">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Issue-alert?">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Transmission-rate">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Surveillance-threshold">
      <value value="50"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
