;; *********************************************************************************
;; *************  Virus Competition Model                    ***********************
;; *************  By Larry Bencivengo                        ***********************
;; *************  May, 2022 - Mercer Island High School  ***********************
;; *********************************************************************************

;; This model simulates virus strains competing to reproduce without killing the host


globals [
  #-cells #-infected-cells #-virus-1 #-virus-2
  #-Infected-1 #-Infected-2 #-Lyso-1 #-Lyso-2
  virus-1-decay-rate virus-2-decay-rate arbitrium-decay-rate
  #-new-virus-1 #-new-virus-2 #-new-arbitrium
]
patches-own [cell? infected-1? infected-2? lytic? arbitrium-absorbed]
breed [viruses virus]
breed [arbitria arbitrium]
turtles-own [decay-rate]
viruses-own [virus-strain]

to setup
  clear-all
  ask patches [set pcolor white] ;; create a white background
  setup-cells                    ;; randomly identify start-#-cells patches as cells
  set #-infected-cells 0
  set #-Infected-1 0
  set #-Lyso-1 0
  set #-Infected-2 0
  set #-Lyso-2 0

  ;; set up virus shapes, starting numbers, decay-rate, etc.
  set #-virus-1 0
  set #-virus-2 0
  set virus-1-decay-rate 20
  set virus-2-decay-rate 20

  ;; generate starting viruses
  set-default-shape viruses "pentagon"
  setup-viruses

  ;; set up arbitrium shape, decay-rate, etc.
  set-default-shape arbitria "dot"
  set arbitrium-decay-rate 30

  reset-ticks
end

  ;;  ****** coded to here ******
  ;;  ***************************

to setup-cells
  ;; generate cells by selecting random patches and setting cell? to true
  ;; repeat until the correct number of cells has been generated
  set #-cells 0
  while [#-cells < start-#-cells] [
    ;; select a random patch - if it is not already a cell, make it a cell
      ask patch random-pxcor random-pycor [if cell? = 0 [
      become-cell
      set #-cells #-cells + 1
      ]
    ]
  ]
end

to become-cell  ;; this routine only called for starting cells
  set cell? 1
  set infected-1? 0          ;; change when infected virus
  set infected-2? 0          ;; change when infected
  set lytic? 0               ;; change when infected by virus in lytic cycle
  set pcolor gray
end

to setup-viruses
  ;; use Virus-1_to_Virus-2_Ratio slider to create the initial virus population
  ;; create virus-1 particles
  set #-new-virus-1 start-#-viruses * (Virus-1_to_Virus-2_Ratio / 100)
  create-viruses #-new-virus-1 [
    set color green
    set size 0.5
    set virus-strain 1
    set decay-rate virus-1-decay-rate
    setxy random-xcor random-ycor
  ]
  ;; update #-virus-1
  set #-virus-1 #-new-virus-1

  ;; create virus-2 particles
  set #-new-virus-2 start-#-viruses - #-new-virus-1
  create-viruses #-new-virus-2 [
    set color blue
    set size 0.5
    set virus-strain 2
    set decay-rate virus-2-decay-rate
    setxy random-xcor random-ycor
  ]
  ;; update #-virus-2
  set #-virus-2 #-new-virus-2

  ;; update #-all-viruses and reset #-new-virus values to 0
  set #-new-virus-1 0
  set #-new-virus-2 0
end


;; to go
;; arbitria molecules and virus particles move randomly
;; arbitria and viruses are "absorbed" by cells as they pass through patches
;; when viruses enter a cell, they will transition to a lytic or lysogenic cycle
;; depending on their lysogenic-probabilty and the level of arbitrium-absorbed
;; when cells accumulate enough viruses [lytic? = # of virus particles], they die ...
;; ... producing a number of new virus particles of the infecting strain equal to lytic?
;; if #-cells is less than 500, all cells with lytic? = 0 have a chance to reproduce
;; free viruses and arbitrium in cells will decay/die at a set rate
;; if #-cells with lytic? = 0 ever falls to zero, the host dies and the simulation ends

to go
  ;; stop when all uninfected cells and viruses dead
  if count patches with [cell? = 1] <= 0 and count viruses <= 0 [stop]
  ;; stop when viruses eradicated and cells take over
  if count viruses <= 0 and #-infected-cells = 0 and count patches with [cell? = 1] >= 500 [stop]
  ask turtles [move]
  ask patches [metabolize]
  add-viruses
  add-arbitrium
  ask patches [reproduce]
  ask patches [update-status]
  tick
end

to move
  ;; molecules and virus particles decay at random intervals
  if random 100 <= decay-rate [
    ifelse is-arbitrium? self [die] [
      if virus-strain = 1 [set #-virus-1 #-virus-1 - 1]
      if virus-strain = 2 [set #-virus-2 #-virus-2 - 1]
      die
    ]
  ]
  ;; arbitrium molecules and viruses diffuse randomly
  right random 30
  left random 30
  forward 1
end

to metabolize  ;; absorb viruses and arbitrium, metabolize arbitrium, generate new viruses, etc.
  ;; only cells metabolize
  if cell? = 1 [
    ;; absorb arbitrium from surroundings - arbitrium builds up and decays over time
    absorb-arbitrium
    if arbitrium-absorbed > 2 [set arbitrium-absorbed arbitrium-absorbed / 2]
    ;; absorb free viruses and become infected
    if infected-1? = 0 and infected-2? = 0 [become-infected]
    ;; if in lytic cycle of infection, produce new virus particles
    if lytic? != 0 [lytic-cycle]
  ]
end

to absorb-arbitrium  ; cell procedure
  let available-arbitrium one-of arbitria-here              ; grab a random arbitrium molecule
  if available-arbitrium != nobody  [
    ask available-arbitrium [die]
    set arbitrium-absorbed arbitrium-absorbed + 1
  ]
  while [count arbitria-here > 1] [
    set available-arbitrium one-of arbitria-here            ; keep absorbing arbitrium
    if available-arbitrium != nobody  [
      ask available-arbitrium [die]
      set arbitrium-absorbed arbitrium-absorbed + 1
    ]
  ]
end

to become-infected  ; cell procedure
  ; grab a random virus molecule
  let available-virus one-of viruses-here
  if available-virus != nobody  [                    ; if viruses present ...
    if [virus-strain] of available-virus = 1 [       ; ... become infected
      set infected-1? 1
      set #-Infected-1 #-Infected-1 + 1
      set #-Lyso-1 #-Lyso-1 + 1
      set #-virus-1 #-virus-1 - 1]
    if [virus-strain] of available-virus = 2 [
      set infected-2? 1
      set #-Infected-2 #-Infected-2 + 1
      set #-Lyso-2 #-Lyso-2 + 1
      set #-virus-2 #-virus-2 - 1]
    ask available-virus [die]                         ; original virus dies
    set #-infected-cells #-infected-cells + 1
    update-infection-status
  ]
end

to lytic-cycle
  ;; if viruses in cell too high, cell dies ...
  ifelse lytic? >= 5 [                  ;; ... and releases free viruses of correct strain
    ;; if infected with strain-1, add new viruses of strain-1
    if infected-1? = 1 [set #-new-virus-1 #-new-virus-1 + lytic?]
    ;; else add new viruses of strain-2
    if infected-2? = 1 [set #-new-virus-2 #-new-virus-2 + lytic?]
    cell-death
    ]
    [ ;; keep producing viruses and arbitrium.
      ;; Rate of virus production inversely proportional to Lysogenic-Probability
      if infected-1? = 1 [
        (ifelse Virus-1-%-Lysogenic <= 25 [set lytic? lytic? + 3]
          Virus-1-%-Lysogenic <= 50 [set lytic? lytic? + 2]
          [set lytic? lytic? + 1])]
      if infected-2? = 1 [
        (ifelse Virus-2-%-Lysogenic <= 25 [set lytic? lytic? + 3]
          Virus-2-%-Lysogenic <= 50 [set lytic? lytic? + 2]
          [set lytic? lytic? + 1])]
      set #-new-arbitrium #-new-arbitrium + lytic?
    ]
end

to update-infection-status
  let chance-lytic 0
  if lytic? = 0 [
    if infected-1? = 1 [
      ifelse arbitrium-absorbed > 1 [
        set chance-lytic (Virus-1-%-Lysogenic * arbitrium-absorbed)]
        [set chance-lytic Virus-1-%-Lysogenic]
      if random 101 > chance-lytic [
        set lytic? 1
        set #-Lyso-1 #-Lyso-1 - 1
      ]
    ]
    if infected-2? = 1 [
      ifelse arbitrium-absorbed > 1 [
        set chance-lytic (Virus-2-%-Lysogenic * arbitrium-absorbed)]
        [set chance-lytic Virus-2-%-Lysogenic]
      if random 101 > chance-lytic [
        set lytic? 1
        set #-Lyso-2 #-Lyso-2 - 1
      ]
    ]
  ]
end

to cell-death
  set cell? 0
  set pcolor white
  if infected-1? = 1 [
    set infected-1? 0
    set #-Infected-1 #-Infected-1 - 1
  ]
  if infected-2? = 1 [
    set infected-2? 0
    set #-Infected-2 #-Infected-2 - 1
  ]
  set lytic? 0
  set arbitrium-absorbed 0
  set #-cells #-cells - 1
  set #-infected-cells #-infected-cells - 1
end

to add-viruses
  ;; create new viruses of strain-1 based on #-new-virus-1
  if #-new-virus-1 > 0 [
    create-viruses #-new-virus-1 [
      set virus-strain 1
      set decay-rate virus-1-decay-rate
      set color green
      set size 0.5
      setxy random-xcor random-ycor
    ]
    set #-virus-1 #-virus-1 + #-new-virus-1
  ]
  ;; create new viruses of strain-2 based on #-new-virus-2
  if #-new-virus-2 > 0 [
    create-viruses #-new-virus-2 [
      set virus-strain 2
      set decay-rate virus-2-decay-rate
      set color blue
      set size 0.5
      setxy random-xcor random-ycor
    ]
    set #-virus-2 #-virus-2 + #-new-virus-2
  ]
  ;; reset values of #-new-virus-1 and #-new-virus-2 for next tick
  set #-new-virus-1 0
  set #-new-virus-2 0
end

to add-arbitrium
  ;; produce new arbitrium molecules based on #-new-arbitrium
  if #-new-arbitrium > 0 [
    create-arbitria #-new-arbitrium [
      set decay-rate arbitrium-decay-rate
      set color black
      set size 0.5
      setxy random-xcor random-ycor
    ]
  ]
  ;; reset value of #-new-arbitrium for next tick
  set #-new-arbitrium 0
end

to reproduce
  ;; only cells reproduce
  if cell? = 1 and lytic? = 0 [     ;; if not in lytic phase, chance to divide
    ;; check for empty adjacent patch
    if any? neighbors with [cell? = 0] [
      ;; probability of dividing decreases as #-cells approaches 500
      if random 500 < ((1 - #-cells / 500) * 100) [
        ;; create a new cell in an adjacent patch
        ;; new cell inherits infection status of original and one half of arbitrium-absorbed
        let virus-1-status infected-1?
        let virus-2-status infected-2?
        set arbitrium-absorbed arbitrium-absorbed / 2
        let amount-arbitrium arbitrium-absorbed
        ;; turn a neighboring patch into a cell
        ask one-of neighbors with [cell? = 0] [
          set cell? 1
          set lytic? 0
          set infected-1? virus-1-status
          set infected-2? virus-2-status
          set arbitrium-absorbed amount-arbitrium
        ]
        ;; update global variables
        set #-cells #-cells + 1
        if infected-1? = 1 or infected-2? = 1 [set #-infected-cells #-infected-cells + 1]
        if infected-1? = 1 [
          set #-Infected-1 #-Infected-1 + 1
          set #-Lyso-1 #-Lyso-1 + 1]
        if infected-2? = 1 [
          set #-Infected-2 #-Infected-2 + 1
          set #-Lyso-2 #-Lyso-2 + 1]
      ]
    ]
  ]
end

to update-status
  ;; update each cell's pcolor to indicate its status:
  ;; healthy = gray, lytic = red, lysogenic = green (virus-1) or blue (virus-2)
  if cell? = 1 [
    update-infection-status
    ifelse lytic? > 0 [
      set pcolor red
    ]
    [(ifelse infected-1? = 1 [
      set pcolor green]
      infected-2? = 1 [
        set pcolor blue]
      [set pcolor gray])
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
256
10
693
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
8
10
71
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
8
47
71
80
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
700
12
872
45
start-#-cells
start-#-cells
0
500
100.0
1
1
NIL
HORIZONTAL

SLIDER
701
86
888
119
Virus-1_to_Virus-2_Ratio
Virus-1_to_Virus-2_Ratio
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
701
124
907
157
Virus-1-%-Lysogenic
Virus-1-%-Lysogenic
0
99
50.0
1
1
NIL
HORIZONTAL

SLIDER
701
162
907
195
Virus-2-%-Lysogenic
Virus-2-%-Lysogenic
0
99
50.0
1
1
NIL
HORIZONTAL

MONITOR
171
86
228
131
arbtm
count arbitria
0
1
11

MONITOR
34
86
96
131
Virus-1
#-Virus-1
0
1
11

MONITOR
99
13
169
58
Total Cells
#-cells
17
1
11

MONITOR
103
86
165
131
Virus-2
#-Virus-2
0
1
11

PLOT
6
134
246
446
Infection Status
Time
concentration
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Host Cells" 1.0 0 -5987164 true "" "plot #-cells"
"Virus-1" 1.0 0 -13840069 true "" "plot #-Virus-1"
"Virus-2" 1.0 0 -13345367 true "" "plot #-Virus-2"
"Arbitrium" 1.0 0 -16777216 true "" "plot count arbitria"
"Infected Cells" 1.0 0 -2674135 true "" "plot #-infected-cells"

MONITOR
174
14
234
59
Infected
#-infected-cells
17
1
11

MONITOR
702
223
788
268
NIL
#-Infected-1
0
1
11

MONITOR
795
223
881
268
NIL
#-Infected-2
0
1
11

MONITOR
724
276
788
321
NIL
#-Lyso-1
0
1
11

MONITOR
796
276
860
321
NIL
#-Lyso-2
0
1
11

SLIDER
702
53
874
86
start-#-viruses
start-#-viruses
10
500
100.0
10
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

A model of different virus strains infecting a host organism.  There are two strains (Virus-1 and Virus-2) competing to become the dominant strain without killing the host.

## HOW IT WORKS

Cells of the host are modeled by patches.  Virus particles are modeled with turtles.

Free virus particles diffuse randomly and can enter an uninfected host cell when they encounter it.  The particles only survive for a short time outside of a host cell.

Once inside the host cell, the virus may enter a lytic (i.e. reproductive) cycle or a lysogenic cycle where it copies its DNA into the host cell's DNA and becomes dormant for a time.  The lytic cycle produces new virus copies and eventually kills the host cell, releasing free virus particles to infect other host cells.  The lysogenic  cycle allows the host cell to survive and reproduce, making copies of the viral DNA in the process.  Eventually the dormant viral genes will "awaken" and enter a lytic cycle, reproducing and killing the host cell.

Different strains of the virus have different probabilities of entering the lytic or lysogenic cycle, and will remain dormant in the lysogenic cycle for various amounts of time.  Healthy, uninfected cells are Gray.  Cells with active (lytic cycle) infections of either strain are Red.  Cells with dormant (lysogenoc cycle) infections are Green or Blue (depending onwhether they are infected by Virus-1 or Virus-2 strains, respectively).

If too many host cells die, the virus will be unable to reproduce and will also die off.  The model attempts to capture the dynamic behavior of virus strains which are competing to reproduce as fast as possible without killing off the host (and each other) in the process.


## HOW TO USE IT

Click Setup, then Go, to see the model run.  You can change several variables using the sliders, including the starting number of host cells, starting number of virus particles, relative amounts of the two strains (Virus-1 vs Virus-2), and the probability that each strain will enter the lysogenic (dormant) life-cycle.

The model will automatically end if either of these conditions are met:
  1.  There are no virus particles of either strain remaining and the cells have reached their maximum value.
  2.  There are no cells and no virus particles of either strain remaining.

Various Monitors allow the user to see important values in the model, as explained below:

Total Cells:  Total # of host cells
Infected: Total # of host cells with infected by either strain of the virus
Virus-1, Virus-2 & arbtrm:  Total # of particles of each strain or # of arbitrium molecules
#-Infected-1, #-Infected-2:  Total # of host cells with an infection by each strain of the virus (this # includes both lytic and lysogenic infections)
#-Lyso-1, #-Lyso-2: Total # of cells with lysogenic infections by each strain (this # excludes lytic infections)



## THINGS TO NOTICE

Changing the probability that a virus strain enters the lysogenic (dormant) cycle will have the biggest effect on viral behavior:  decreasing the probability will produce more active infections and tend to kill off the host cells more quickly, while increasing the probability will produce fewer active infections and allow host cells to survive and reproduce (perhaps with viral DNA stowing away inside them!).  

Changing the starting number of host cells, number of virus particles or ratio of the two strains will primarily alter the chance that the host cells die off or survive.

Can you figure out what is happening with the mysterious *arbitrium*?

## THINGS TO TRY

Which settings will tend to kill off the host most rapidly?  Which ones will allow the host cells and virus to survive in a sort of equilibrium?  What advantage does the virus gain by entering the lysogenic cycle?  What happens when the two viral strains have very different probabilities of entering the lysogenic phase?  Is this outcome affected by the starting numbers of cells and/or virus particles?  The relative amounts of the two strains at the start?

Try stopping the model at various times to see how the levels of arbitrium are related to the number of virus particles.  How are the level of arbitrium and the number of cells with lysogenic infections related?

Read the following article from Nature which is the basis for this model: https://www.nature.com/articles/d41586-019-01880-6

It will help you to understand the role of arbitrium in viral behavior.  Is there a way to set up the two strains of the virus to simulate how a strain could "cheat" by ignoring the arbitrium signal, thereby gaining a competitive advantage over theo other strain?  Over what range of settings is it possible for this "strategy" to be successful?  Over what range is it doomed to failure?

## EXTENDING THE MODEL

The model could be altered to allow multiple virus strains to infect the same host cell.  Additional strains of the virus could be introduced.

The host cells currently have no defenses against viral infection; the behavior of the model would be dramatically altered if host cells could fight off infection by virus particles.

It could be interesting to explore an evolutionary model by giving the viruses genes which can mutate as the host cells produce new virus particles during lytic cycles.  Thus, new strains of the virus would tend to emerge spntaneousy and begin competing with each other.  Some interesting "genes" might control the lysogenic probability, replication speed, arbitrium sensitivity, etc.  

## NETLOGO FEATURES

One interesting feature of this model is the use of patches to represent cells and turtles to represent molecules/virus particle which are diffusing.

## RELATED MODELS

This model is based upon an earlier model called "Cells in a Petri Dish".  

## CREDITS AND REFERENCES

Larry Bencivengo - Mercer Island High School (2022).

Dolgin, Elie (2019). "The Secret Social Lives of Viruses" in Nature, 18 June 2019.
https://www.nature.com/articles/d41586-019-01880-6
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

hex
false
0
Polygon -7500403 true true 0 150 75 30 225 30 300 150 225 270 75 270

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

molecule carbon dioxide
true
0
Circle -7500403 true true 165 45 90
Circle -16777216 false false 165 45 90
Circle -16777216 true false 90 75 120
Circle -16777216 false false 90 75 120
Circle -7500403 true true 30 120 120
Circle -16777216 false false 30 120 120

molecule oxygen
true
0
Circle -7500403 true true 120 75 150
Circle -16777216 false false 120 75 150
Circle -7500403 true true 30 75 150
Circle -16777216 false false 30 75 150

molecule water
true
0
Circle -1 true false 183 63 84
Circle -16777216 false false 183 63 84
Circle -7500403 true true 75 75 150
Circle -16777216 false false 75 75 150
Circle -1 true false 33 63 84
Circle -16777216 false false 33 63 84

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
