;; Cellular Respiration: Mitochondrial Electron Transport Chain
;;
;; Coded in 2019 by Lin Xiang (lin.xiang@uky.edu), based on the ElectronTransport model initially developed by Steven Brewer.
;;
;; Brewer, S.D. (2004).  Electron Transport: Simulating Proton Gradient, ADP-ATP, and NADH-NAD+ Interactions.
;; http://bcrc.bio.umass.edu/netlogo/models/ElectronTransport/Biology Computer Resource Center. (lxiang75@gmail.com; lin.xiang@uky.edu)
;;
;; If you mention this model in a publication, we ask that you include the citations below.
;;
;; Xiang, L. (2019). Cellular Respiration: Mitochondrial Electron Transport Chain. Department of STEM Education, University of Kentucky, Lexington, KY.
;;
;;-----------------------------------------
;;CREATIVE COMMONS LICENSE
;;This code is distributed by Lin Xiang under a Creative Commons License:
;;Attribution-ShareAlike 4.0 International (CC BY-SA 4.0)
;;https://creativecommons.org/licenses/by-sa/4.0/
;;
;;-----------------------------------------

breed [enzymes enzyme]
breed [hs h]
breed [adps adp]
breed [nads nad]
breed [water a-water]
breed [oxygens oxygen]
breed [lipids lipid]
breed [poison1s poison1]
breed [poison2s poison2]

patches-own[protein]
hs-own [pass]

globals [n n1 atp]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Main Procedures (SETUP and GO)                                    ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
 ca

 set atp 100   ;have some ATPs to survive

 setup-membrane
 setup-NADH
 setup-oxygen
 setup-protons

reset-ticks
end


to go
tick

  nad-nadh-action
  H-movement
  H-gradient-inhabitor-uncoupler
  ATP-water-formation
  oxygen-water-move
  NADH-formation
  ATP-consumption
  ;no-display
  plot-levels
  ;display

  if ATP <= 0 [user-message "The cell runs our of ATPs and dies." stop]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Sub-procedures of SETUP                                           ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup-membrane

;make the inner membrane
  ask patches with [abs pycor <= 3] [set pcolor 49 ]

; make boundary for the system
  ask patches with [abs pxcor = max-pxcor or abs pycor = max-pycor] [set pcolor 2]

  ;add enzymes into the membrane
  ask patches with [pcolor = 49] [

    if pxcor >= -10 and pxcor <= -8 or
       pxcor >= -2 and pxcor <= 0 or
       pxcor >= 6 and pxcor <= 8
    [set pcolor 115 set protein 2]     ;set up ETS.

    if pxcor = -6 or
       pxcor = 2 or
       pxcor = 10
    [set pcolor 72 set protein 1]      ;set up ATP synthase

    if pxcor = -12 or
       pxcor = -4 or
       pxcor = 4 or
       pxcor = 12
    [set pcolor 138 set protein 3]      ;set up leak proteins (such as ANT or UCP)

  ]

  ask patches with [ pycor = 3 and pcolor = 49] [sprout-lipids 1 [set color 7 set shape "lipid" set size 4 set heading 0 set ycor ycor - 1.5]]
  ask patches with [ pycor = -3 and pcolor = 49] [sprout-lipids 1 [set color 7 set shape "lipid" set size 4 set heading 180 set ycor ycor + 1.5]]  ;setup the bilayer

 end


to setup-protons
  create-hs 30
  [set color 25
    set shape "proton1"
    set size 0.6
  setup-position]
end

to setup-oxygen
  create-oxygens 5 + random 3 [set size 2 set color white set shape "oxygen"  setup-position-1]
end

to setup-position     ;random positions in both up and bottom black areas
 setxy random-xcor random-ycor
  if pcolor != 0 [setup-position]
end

to setup-position-1       ;random positions in the bottom black area
 setxy random-xcor random-ycor
  if pcolor != 0 or ycor >= -4 [setup-position-1]
end

to setup-NADH       ;make 100 NADH molecules at the beginning
  create-nads 100
  [set color 25
    set shape "nadh"
    set size 2.5
    setup-position-1]
end

to setup-poison1s
  ask patches with [pcolor = 113 and pycor = -3]
   [sprout-poison1s 1[
    set size 1
    set color yellow
    set shape "p-1"
  ]]
end

to setup-poison2s
  create-poison2s 8
   [set size 1
    set color yellow
    set shape "p-2"
    setup-position]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Sub-procedures of GO                                              ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to nad-nadh-action
  ask nads [right random 90 left random 90   ; ask NADHs
    ifelse [pcolor] of patch-at dx dy != 0 [ifelse [pcolor] of patch-at dx dy != 115 [set heading heading - 180] ; Turn back if approching to a structure but not an ETC,
      [if shape = "nadh" [set shape "nad" set color 3 set size 2 hatch-hs 1 [set color 25 set shape "proton1" set size 0.6 fd 1  set pass 0]]]]  ;if approaching to an ETC, change to NAD+ and release one H+
    [fd 1]
  ]
end

to H-gradient-inhabitor-uncoupler        ;;Activate or deactivate ETC based on H+ gradient, inhabitor and uncouplers
  ask patches with [protein = 1] [          ;ask ATP synthase
    ifelse Poison_ATP_synthase = true             ;If poisoned
      [set pcolor 72]                       ;change pcolor to deactivate ATP synthase
      [ifelse count hs with [ycor > 3] - count hs with [ycor < -3] >= 10 [set pcolor 75][set pcolor 72]]   ; if not poisoned, stay activated if H+ gradient is established, deactivated if H+ gradient is too low.
      ]

  ask patches with [protein = 2] [          ;ask ETC
    ifelse Poison_ETC = true or block-oxygen = true   ;If poisoned or oxygen is blocked
    [set pcolor 113]                        ;change pcolor to deactivate ETC
    [ifelse count hs with [ycor > 3] / (1 + count hs with [ycor < -3]) >= 200 [set pcolor 113][set pcolor 115]]]; if not poisoned, stay activated if H+ gradient is not too high, deactivated if H+ gradient is too high.

  ask patches with [protein = 3] [          ;ask leak proteins
    ifelse Uncouplers = true                ;If uncoupled
      [set pcolor 137]                      ;change pcolor to activate uncouplers
      [set pcolor 138]                      ; change color to deavtiate uncouplers
      ]

  oxygen-factor

end

to H-movement
  ask hs
  [
    if pcolor = 0
      [right random 90 left random 90   ;wiggle
         if [pcolor] of patch-at dx dy = 0 [fd 0.3]   ;If nothing is aheadm, move forward
         if [pcolor] of patch-at dx dy  = 75 [ifelse heading > 90 or heading < 270  [fd 1][set heading heading - 180]]  ;if an H+ gets close to an ATP synthase and if it is in the intermembrane space, bound with the ATP synthase
         if [pcolor] of patch-at dx dy = 115 [ifelse heading < 90 or heading > -90  [fd 1][set heading heading - 180]] ;if an H+ gets close to an ETC and it is in the matrix, bound with ETC complex
         if [pcolor] of patch-at dx dy = 137  [if heading > 90 or heading < 270  [ifelse uncouplers = true [fd 1][set heading heading - 180]]]  ;if an H+ gets close to an leak protein and it is in the intermembrane space, if uncouples are true, bound to the protein.
     ]

    if pcolor = 75 [set heading 180 fd 0.1       ;if a H+ has entered in an ATP synthase, move down into the matrix
      if ycor < -3 and pass = 1 [set pass 2]]    ;if a H+ go DOWN through the inner membrane, label its pass as 2


    if pcolor = 115 [set heading 0 fd 0.1          ;if a H+ has entered in an ETC complex, move up to the inter membrane space
      if ycor > 3 and pass = 0 [set pass 1]]       ;if a H+ go UP through the inner membrane, label its pass as 1


    if pcolor = 137 [set heading 180 fd 0.05      ;if a H+ has entered the bilayer, move down into the matrix
      if ycor < -3 and pass = 1 [set pass 2]]    ;if a H+ go DOWN through the inner membrane, label its pass as 2

    if ycor > 0 and pass = 2 [set pass 1]   ;some Hs may go back to inter membrane space, label the pass as 1 so it will not form water in the wrong place.
  ]

end


to ATP-water-formation
  ask hs with [pass = 2 and pcolor = 75] [ if count oxygens > 0   ;ask Hs that pass through ATP synthase, if oxygens are present
    [hatch-water 1 [set shape "water" set color red set size 1.5 ] ;mostly of Hs form water molecule. Note Hs assumption and water formation are not in the ratio of 4:1.
      set atp atp + 3 die  ]]      ;Each H makes 3 ATPs

  ask hs with [pass = 2 and pcolor = 137] [ if count oxygens > 0   ;ask Hs that pass uncouplers, if oxygens are present
    [hatch-water 1 [set shape "water" set color red set size 1.5 ] ;mostly of Hs form water molecule. Note Hs assumption and water formation are not in the ratio of 4:1.
       die ]]      ;No ATPs are produced
end

to oxygen-water-move       ;make water and oxygen only move within the matrix
  ask water [right random 90 left random 90
  ifelse [pcolor] of patch-at dx dy != 0
    [set heading heading - 180] [fd 0.3]]

  ask water [if [pcolor] of patch-at dx dy = 2 [die]]    ;if water gets to the boundary, disappears

  ask oxygens [right random 90 left random 90
  ifelse [pcolor] of patch-at dx dy != 0
    [set heading heading - 180] [fd 0.3]]

end

to NADH-formation    ;reform NADH at an ajustable rate
  ask nads with [color = 3]
  [ if random 1000 < formation-of-NADH [
    ;if random-float 1 < (formation-of-NADH * 0.005) [
    set color 25
    set shape "nadh"
    set size 2.5]]
end


to ATP-consumption
 if random 100 < ATP-consumption-rate [set atp atp - 1]
end


to  oxygen-factor

  if block-oxygen = true [ask oxygens [die]]
  if block-oxygen = false [ if count oxygens <= 0 [create-oxygens 5 + random 3 [set size 2 set color white set shape "oxygen"  setup-position-1]]]

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Supporting-procedures                                             ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to plot-levels ;; this creates creates the bar graph
    set-current-plot "Proton Gradient"
    clear-plot
    plot-pen-down
    set-current-plot-pen "matrix"
    plotxy 1 count hs with [ycor <= -3]
    set-current-plot-pen "intermembrane"
    plotxy 2 count hs with [ycor >= 3]

end


; Copyright 2019 by Lin Xiang.  All rights reserved.
;
; This model is modified based on the ElectronTransport model initially developed by Steven Brewer.
;
; Brewer, S.D. (2004).  Electron Transport: Simulating Proton Gradient,
; ADP-ATP, and NADH-NAD+ Interactions.
; http://bcrc.bio.umass.edu/netlogo/models/ElectronTransport/
; Biology Computer Resource Center
@#$#@#$#@
GRAPHICS-WINDOW
19
10
549
433
-1
-1
18.0
1
10
1
1
1
0
1
1
1
-14
14
-11
11
0
0
1
ticks
30.0

BUTTON
559
13
654
73
Setup/Reset
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
559
78
654
138
Run/Pause
go\nif Investigation-Time <= ticks [user-message \"The simulation has run the defined ticks. You may increase the time to continue running the simulation.\"  stop]\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
660
78
779
138
Run a tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
560
153
752
186
Formation-of-NADH
Formation-of-NADH
0
10
1.0
1
1
NIL
HORIZONTAL

PLOT
372
443
687
624
ATP Yields
Ticks
# of ATPs
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"ATP" 1.0 0 -2674135 true "" "plot atp"

INPUTBOX
660
14
779
74
Investigation-Time
1000000.0
1
0
Number

SLIDER
560
193
752
226
ATP-consumption-rate
ATP-consumption-rate
0
50
27.0
1
1
NIL
HORIZONTAL

MONITOR
693
444
779
489
ATP
atp
0
1
11

SWITCH
560
238
732
271
Block-oxygen
Block-oxygen
1
1
-1000

SWITCH
560
313
732
346
Poison_ETC
Poison_ETC
1
1
-1000

SWITCH
560
352
732
385
Poison_ATP_synthase
Poison_ATP_synthase
1
1
-1000

SWITCH
560
275
732
308
Uncouplers
Uncouplers
1
1
-1000

PLOT
19
443
361
625
Proton Gradient
NIL
NIL
0.0
4.0
0.0
10.0
true
true
"" ""
PENS
"matrix" 1.0 1 -16777216 true "" "plotxy 1 count hs with [ycor <= -3]"
"intermembrane" 1.0 1 -7500403 true "" "plotxy 2 count hs with [ycor >= 3]"

TEXTBOX
39
636
448
660
Tip: Run the model at a slower speed to get a better visual result.
11
105.0
1

@#$#@#$#@
## WHAT IS IT?

This model simulates the processes of proton transport and ATP formation in a mitochondrion. Students may observe the processes and explore the impacts of ETC inhibitors, ATP synthase inhibitors, uncoupling proteins, or the absence of oxygen on these processes. This model is modified from a NetLogo simulation developed by Steven Brewer (2004). Detailed membrane structures, rules of proton movement and regulation, and oxygen impacts are constructed in this model to provide a powerful dynamic visualization to support NGSS-aligned active learning at 6-12 levels.

## HOW IT WORKS

<b>Components---</b>

<b>ETCs</B>: purple patch areas
<b>ATP synthases</B>: green patch areas
<b>uncoupling proteins</B>: pink patch area
<b>lipid bilayer</B>:yellow patch area with lipids

<b>NADH</B>: orange flat hexagons
<b>NAD</B>: gray flat hollow hexagons
<b>Protons</B>: orange circles containing a plus
<b>Oxygens</B>: white double circles
<b>Water</B>: white and red molecules
<b>ATP</b>: invisible in the model but can be tracked in the plot of "ATP Yields".

<b>Rules:---</b>

* <b>NADH</B> molecules in the matrix pass a proton (H+) to an <b>electron transport chain (ETC)</b>. 

* A functional <b>ETCs</B> transports H+ into the inter membrane space. An <b>ETCs</B> will not function if 1) it is poisoned (e.g., by an inhabitor), 2) oxygen is blocked, or 3) H+ gradient is too high (arbitrarily set as 200 times higher in this simulation). 

* A functional <b>ATP synthase</b> allows H+ to move into matrix. If oxygens are present, water and ATP will be produced. An <b>ATP synthase</b> will not function if 1) it is inhabited, or 2) H+ gradient is too low (there must be ten more protons in intermembrane space than in the matrix).

* An **uncoupling protein** allows H+ to move into the matrix if activated. If oxygens are present, water will be produced, but no ATP will be produced.


## HOW TO USE IT

1. Click on the "setup/reset" button to initiate or reset the model. 
2. Put down the time you want to model to run. For a preliminary investigation, put down a big number (e.g., 1000000) to get a sufficient observation time.
3. Click on the "Run/pause" button to run or pause the model. Click on the "Run a tick" button to run the model for just one tick. 
4. Use switches to explore/test the impacts of ETC inhibitors, ATP synthase inhibitors, uncoupling proteins, or absence of oxygen on the production of ATP.
5. Use sliders to adjust the formation rate of NADH and the consumption rate of ATP.

**Run the model at a lower speed to get a better visual effect.**

## THINGS TO NOTICE AND TRY

* When using the model, pay attention to how protons move. Can you infer which side of the model represents the matrix and which side represents the intermembrane space? What is the evidence in the model?

* What happens when oxygen is blocked? The absence of oxygen first affect the ETCs or ATP synthases? What is the evidence in the model?

* What happens when ETCs are poisoned? How does it affect the proton gradient and formation of ATP? Will it also affect ATP synthases? How and what is your evidence?

* What happens when ATP synthases are poisoned? How does it affect the proton gradient and formation of ATP? Will it also affect ETCs? How and what is your evidence?

* What happens when uncoupling proteins are activated? How does it affect the proton gradient and formation of ATP? Will it also affect ETCs and/or ATP synthases? How and what is your evidence?

* How do NADH affect the proton gradient and formation of ATP? What is your evidence?

* What are the real-life examples of the simulated scenarios above?


## RELATED MODELS

Check [3dsciencemodeling.com](http://3dsciencemodeling.com) for related models and other resources.


## CREDITS AND REFERENCES

This model is made by Dr. Lin Xiang at the University of Kentucky.If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2019). Cellular Respiration: Mitochondrial Electron Transport Chain. Department of STEM Education, University of Kentucky, Lexington, KY.

This model is modified based on the ElectronTransport model initially developed by Steven Brewer.

Brewer, S.D. (2004).  Electron Transport: Simulating Proton Gradient, ADP-ATP, and NADH-NAD+ Interactions. http://bcrc.bio.umass.edu/netlogo/models/ElectronTransport/Biology Computer Resource Center.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

adp
true
0
Circle -7500403 true true 201 127 48
Circle -7500403 true true 152 127 48
Polygon -7500403 true true 150 150 120 195 60 195 30 150 90 105

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

atp
true
0
Circle -7500403 true true 250 126 48
Circle -7500403 true true 201 127 48
Circle -7500403 true true 152 127 48
Polygon -7500403 true true 150 150 120 195 60 195 30 150 90 105

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

enzyme1
true
0
Rectangle -7500403 true true 75 30 225 270

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

lipid
true
0
Circle -7500403 false true 113 1 74
Line -7500403 true 165 75 165 105
Line -7500403 true 135 75 135 105
Line -7500403 true 135 105 150 120
Line -7500403 true 150 120 135 135
Line -7500403 true 135 135 150 150
Line -7500403 true 135 165 150 180
Line -7500403 true 135 195 150 210
Line -7500403 true 135 225 150 240
Line -7500403 true 135 255 150 270
Line -7500403 true 135 285 150 300
Line -7500403 true 165 105 180 120
Line -7500403 true 165 135 180 150
Line -7500403 true 165 165 180 180
Line -7500403 true 165 195 180 210
Line -7500403 true 165 225 180 240
Line -7500403 true 165 255 180 270
Line -7500403 true 150 150 135 165
Line -7500403 true 180 120 165 135
Line -7500403 true 180 150 165 165
Line -7500403 true 180 180 165 195
Line -7500403 true 150 180 135 195
Line -7500403 true 180 210 165 225
Line -7500403 true 150 210 135 225
Line -7500403 true 150 240 135 255
Line -7500403 true 150 270 135 285
Line -7500403 true 180 240 165 255

lipids
true
0
Circle -7500403 false true 128 1 44
Line -7500403 true 145 45 145 236
Line -7500403 true 165 40 166 269

nad
true
0
Polygon -7500403 false true 30 150 75 90 165 90 210 150 165 210 75 210

nadh
true
0
Polygon -7500403 true true 75 90 30 150 75 210 165 210 210 150 165 90
Circle -16777216 false false 117 117 66
Line -16777216 false 135 150 165 150
Line -16777216 false 150 135 150 165

nadh-1
true
0
Polygon -7500403 true true 75 90 30 150 75 210 165 210 210 150 165 90
Circle -7500403 false true 207 117 66
Line -7500403 true 225 150 255 150
Line -7500403 true 240 135 240 165

oxygen
true
0
Circle -7500403 true true 135 90 120
Circle -7500403 true true 45 90 120

p-1
true
0
Rectangle -7500403 true true 15 120 285 180
Circle -7500403 true true 75 30 150

p-2
true
0
Polygon -7500403 true true 0 135 45 90 105 135 150 90 195 135 240 90 300 135 300 165 240 120 195 165 150 120 105 165 45 120 0 165

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

proton
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Line -7500403 true 150 75 150 225
Line -7500403 true 75 150 225 150

proton1
true
0
Circle -16777216 true false 13 13 272
Circle -7500403 false true 15 15 270
Line -7500403 true 150 90 150 210
Line -7500403 true 90 150 210 150

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

water
true
0
Circle -1 true false 65 35 170
Circle -7500403 true true 207 131 67
Circle -7500403 true true 26 131 67

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
