extensions [rnd];This extension can be dowloaded at https://github.com/NetLogo/Rnd-Extension

globals [
  fizz ; measures the frequency of attitudes that change at each step
  left-freq ;frequency of negative attitudes
  right-freq ; frequency of positive attitudes
  ]

turtles-own [
  old-attitude
  new-attitude
  id-neighbors
  ]

to setup
  clear-all
  ask patches [set pcolor gray]
  set-default-shape turtles "square"
  make-turtle-distribution
  ask turtles [
    set size 1.2
    set old-attitude new-attitude
    set color 5 - ((old-attitude / max-entrench) * 4.9)
  ] ;; in order to do synchronous updating, we need to delay when agents update their attitudes, so we separate attitudes into new and old
  set fizz 0
  measure-frequency-over-time

  reset-ticks
end

to go
  ask turtles [ local-interactions ]
  set fizz (count turtles with [old-attitude != new-attitude]) / count turtles
  ask turtles [
   set old-attitude new-attitude
   set color 5 - ((old-attitude / max-entrench) * 4.9)
  ]
  measure-frequency-over-time
  tick
end

to local-interactions
  let j nobody ; j will be the other agent, which we initially set as nobody
  ifelse influence = "max-inf" or influence = "min-inf" [
    if influence = "max-inf" [
      set j max-one-of turtles-on neighbors [abs old-attitude] ]
    if influence = "min-inf" [
      set j min-one-of turtles-on neighbors [abs old-attitude] ]
  ]
  [
    let localsum 0 ; This will be sum of the neighbour attitudes, weighted by the influence function.
    let pairlist [] ; this is going to be the list of pairs with a turtle first and a probability second
    ask turtles-on neighbors [
      set localsum localsum + abs influence-function [old-attitude] of self ]
    ask turtles-on neighbors [
      let prob 0
      if localsum != 0 [set prob (abs influence-function [old-attitude] of self) / localsum]
      let pair []
      set pair lput self pair
      set pair lput prob pair
      set pairlist lput pair pairlist
    ]
    set j first rnd:weighted-one-of pairlist [last ?]
  ]
  if is-turtle? j [ ;it is possible that no agent ends up getting picked in the line above, so this checks that someone has
    ifelse random-float 1 < opinion-amplification [
      if [old-attitude] of j > 0 [
        if abs old-attitude < max-entrench [
          ifelse old-attitude = -1 [set new-attitude 1] [ set new-attitude old-attitude + 1 ]]
      ]
      if [old-attitude] of j < 0 [
        if abs old-attitude < max-entrench [
          ifelse old-attitude = 1 [set new-attitude -1] [set new-attitude old-attitude - 1 ]]
      ]]
    [
      if [old-attitude] of j > old-attitude [
        ifelse old-attitude = -1 [set new-attitude 1] [ set new-attitude old-attitude + 1]]
      if [old-attitude] of j < old-attitude [
        ifelse old-attitude = 1 [ set new-attitude -1][set new-attitude old-attitude - 1 ]]
    ]
  ]
end

to-report influence-function [item1]
  if influence = "linear" [
    ifelse item1 = 0 [report 1] [report item1] ]  ; this is the linear update function
  if influence = "square" [
    ifelse item1 = 0 [report 1] [report (item1 ^ 2)]] ; this weighs extreme attitudes more than moderate ones
  if influence = "uniform" [
    if item1 < 0 [report -1]
    if item1 > 0 [report 1] ]
  if influence = "co-linear" [
    ifelse item1 = 0 [report 1] [report (max-entrench + 1 - abs item1) ] ]
  if influence = "co-square" [
    ifelse item1 = 0 [report 1] [report (max-entrench + 1 - abs item1) ^ 2 ] ]
end

to make-turtle-distribution
ifelse rnd-initial [
    ask patches [;; every patch will either have a positive or negative attitude placed on them. The probability of one over the other is set by polluter-to-nonpolluter-ratio
    ifelse random-float 1 < positive-to-negative-ratio [
      sprout 1 [
        set new-attitude (random init-attitude) + 1
      ]
    ]
    [
      sprout 1 [
        set new-attitude 0 - (random init-attitude) - 1
      ]
    ]
  ]
][;;otherwise, the positive attitudes are arranged to form a circle, with a radius set by radius-size
ask patches [
    ifelse (abs pxcor) ^ 2 + (abs pycor) ^ 2 <= radius-size ^ 2 [
      sprout 1 [
        set new-attitude max-entrench
      ]
    ]
    [
      sprout 1 [
        set new-attitude (0 - max-entrench)
      ]
    ]
  ]
]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; STATISTICS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to measure-frequency-over-time
  set left-freq count turtles with [old-attitude <= -1] / count turtles
  set right-freq count turtles with [old-attitude >= 1] / count turtles
end

;; Copyright Bert Baumgaertner
;; See info tab for credits
@#$#@#$#@
GRAPHICS-WINDOW
370
24
784
459
50
50
4.0
1
10
1
1
1
0
1
1
1
-50
50
-50
50
1
1
1
ticks
30.0

BUTTON
13
65
68
98
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
76
65
131
98
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
208
105
344
138
init-attitude
init-attitude
0
max-entrench
2
1
1
NIL
HORIZONTAL

SLIDER
208
64
344
97
max-entrench
max-entrench
0
10
3
1
1
NIL
HORIZONTAL

PLOT
13
370
360
505
Attitude Distribution
NIL
NIL
-10.0
10.0
0.0
500.0
true
false
"set-plot-x-range (-1 - max-entrench) max-entrench + 1\nset-plot-y-range 0 count turtles / 4\nset-histogram-num-bars (2 * max-entrench) + 1" ""
PENS
"default" 1.0 1 -2674135 true "" "histogram [old-attitude] of turtles"

BUTTON
141
65
196
98
Step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
12
10
150
55
Influence
Influence
"uniform" "linear" "square" "co-linear" "co-square"
0

SLIDER
158
10
348
43
opinion-amplification
opinion-amplification
0
1
0.1
0.01
1
NIL
HORIZONTAL

SLIDER
13
193
185
226
radius-size
radius-size
0
20
20
1
1
NIL
HORIZONTAL

SWITCH
13
110
132
143
rnd-initial
rnd-initial
0
1
-1000

SLIDER
13
151
210
184
positive-to-negative-ratio
positive-to-negative-ratio
0
1
0.5
0.01
1
NIL
HORIZONTAL

PLOT
13
236
360
364
Frequencies across time
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Left" 1.0 0 -955883 true "" "plot left-freq"
"Right" 1.0 0 -11783835 true "" "plot right-freq"
"Fizz" 1.0 0 -16777216 true "" "plot fizz"

@#$#@#$#@
## WHAT IS IT?

This model shows the spatial dynamics of attitude formation.

## HOW IT WORKS

Agents select a neighbor to interact with according to an influence function. If that neighbor has the same opinion (i.e., both attitudes are positive or both are negative) then there is some probability that the focal agent "amplifies" their opinion by changing their attitude to become more extreme. This creates clusters that increase in size and have boundaries with different types of behaviors (see section on THINGS TO NOTICE).

## HOW TO USE IT

There are two main parameters that determine how attitudes will change over time. The "Influence" button controls how focal agents select a neighbor. If "uniform" is selected, then each neighbor has the same chance of being picked for an interaction, regardless of their attitude. The options "linear" and "quadratic" bias those neighbors with more extreme opinions (higher absolute-valued attitudes), while options "co-linear" and "co-quadratic" bias more moderate opinions (attitudes closer to zero). Note that there are no zero attitudes.

The second main parameter is "opinion-amplification". If a focal agent shares the same opinion type of the neighbor that's been selected for interaction, then with probability p the focal individual adopts a more extreme attitude, where p is specified by the value of opinion-amplification.

The model can be initialized with respect to both the distribution of attitudes on a spectrum and their spatial location. The size of the attitude spectrum is set by "max-entrench" - it specifies what the most extreme attitudes can be. E.g., if "max-entrench" is set to 3, then the possible attitudes are {-3,-2,-1,1,2,3}. "init-attitude" specifies the range of attitudes that the simulation will start with, and "positive-to-negative-ratio" determines the frequency of positive attitudes relative to negative attitudes.

If "rnd-initial" is ON, then attitudes will be randomly assigned in space. If it is OFF, then all the positive attitudes will form a circle in the middle. The size of the circle is controlled by "radius-size".


## THINGS TO NOTICE

When there is enough opinion amplification, boundaries between clusters exhibit surface tension and straighten out with what appears to be motion by mean curvature. If one type of opinion is surrounded by the second type, the first type will tend to decrease in number and the cluster is "swallowed up". If bands form with straight boundaries, then both types of opinion will coexist for a very long time.


## THINGS TO TRY

If you select "linear" or "quadratic" influence, then the surface tension effect is increased. If you select "co-linear" influence, then you need to increase the amplification level to produce the surface tension effect, and even more so if you pick "co-quadratic" influence.

Turn off "rnd-initial" so that the simulation starts with a well defined boundary in the form of a circle. Notice that for uniform influence, you only need a little bit of amplification (e.g., 0.05) to keep the form of the circle. Also notice that the circle slowly decreases in size.


## EXTENDING THE MODEL

Interactions are local in this model. Add code for "global" interactions, that is, create a new slider such that with some probability a focal agent will select a person from the population at random. As this probability is increased, the system becomes noisier and consensus is reached more quickly (i.e., one opinion type wins).

## NETLOGO FEATURES

This model makes use of the "rnd" extension. The influence function is a weighted probability. When it is "linear" then the weights of more extreme opinions grow linearly with the absolute value of the attitude. When it is "quadratic" then the weights grow by the square.


## CREDITS AND REFERENCES

Copyright 2015 Bert Baumgaertner.

Thanks to collaborators Steve Krone and Rebecca Tyson.

This project was funded by CLASS Excellence in Teaching the Humanities Endowment and NIH Grant P20GM104420.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -7500403 true true 135 285 195 285 270 90 30 90 105 285
Polygon -7500403 true true 270 90 225 15 180 90
Polygon -7500403 true true 30 90 75 15 120 90
Circle -1 true false 183 138 24
Circle -1 true false 93 138 24

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>positive-ratio</metric>
    <metric>negative-ratio</metric>
    <metric>zero-ratio</metric>
    <metric>right-polarization</metric>
    <metric>left-polarization</metric>
    <metric>right-similarity</metric>
    <metric>left-similarity</metric>
    <enumeratedValueSet variable="min-carbon-contribution">
      <value value="1.4E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transformative-changes">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-pollution-level">
      <value value="310"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="armageddon">
      <value value="700"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="carbon-sequester">
      <value value="0.5"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-fraction" first="0" step="0.05" last="1"/>
    <enumeratedValueSet variable="max-carbon-contribution">
      <value value="0.00193"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.05" last="1"/>
    <enumeratedValueSet variable="delay-armageddon">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Eatingcircles" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="radius-size" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;linear&quot;"/>
      <value value="&quot;exp&quot;"/>
      <value value="&quot;equal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="EatingcirclesEqualWithAmp" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="radius-size" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.15" last="0.9"/>
  </experiment>
  <experiment name="EatingcirclesLinearWithAmp" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="15000"/>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="radius-size" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;linear&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.15" last="0.9"/>
  </experiment>
  <experiment name="EatingcirclesExpWithAmp" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="20000"/>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="radius-size" first="1" step="1" last="10"/>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;exp&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.15" last="0.9"/>
  </experiment>
  <experiment name="Fizz" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="500"/>
    <metric>fizz</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;exp&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FizzNoAmp" repetitions="25" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;exp&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FizzLowAmp" repetitions="25" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0"/>
      <value value="0.05"/>
      <value value="0.1"/>
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;exp&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TippingPointsEqual" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0.1"/>
      <value value="0.3"/>
      <value value="0.5"/>
      <value value="0.7"/>
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="0.75"/>
      <value value="0.8"/>
      <value value="0.85"/>
      <value value="0.9"/>
      <value value="0.95"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TippingPointsLinear" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0.1"/>
      <value value="0.3"/>
      <value value="0.5"/>
      <value value="0.7"/>
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;linear&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="0.75"/>
      <value value="0.8"/>
      <value value="0.85"/>
      <value value="0.9"/>
      <value value="0.95"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TippingPointsCubic" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0.1"/>
      <value value="0.3"/>
      <value value="0.5"/>
      <value value="0.7"/>
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;exp&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="0.75"/>
      <value value="0.8"/>
      <value value="0.85"/>
      <value value="0.9"/>
      <value value="0.95"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FinalLocalLowAmp" repetitions="50" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <metric>positive-ratio</metric>
    <metric>negative-ratio</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0"/>
      <value value="0.025"/>
      <value value="0.05"/>
      <value value="0.075"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FinalLocalCoInf" repetitions="25" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>right-polarization</metric>
    <metric>right-sd</metric>
    <metric>left-polarization</metric>
    <metric>left-sd</metric>
    <metric>positive-ratio</metric>
    <metric>negative-ratio</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;co-linear&quot;"/>
      <value value="&quot;co-square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FinalLocalFreq" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;square&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;equal&quot;"/>
      <value value="&quot;co-linear&quot;"/>
      <value value="&quot;co-square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="FinalLocalFreqRelocate" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="250"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;square&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;equal&quot;"/>
      <value value="&quot;co-linear&quot;"/>
      <value value="&quot;co-square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Voter11" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="200"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.05" last="0.5"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;square&quot;"/>
      <value value="&quot;linear&quot;"/>
      <value value="&quot;equal&quot;"/>
      <value value="&quot;co-linear&quot;"/>
      <value value="&quot;co-square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="RelocationAmp0InfSq" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="200"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocate-freq">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bin-p">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-dist">
      <value value="&quot;Uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-to-relocate">
      <value value="0"/>
      <value value="10"/>
      <value value="25"/>
      <value value="50"/>
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="RelocationAmp05InfEq" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="200"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocate-freq">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bin-p">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-dist">
      <value value="&quot;Uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-to-relocate">
      <value value="0"/>
      <value value="10"/>
      <value value="25"/>
      <value value="50"/>
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="RelocationAmp02InfCoLi" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="200"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocate-freq">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bin-p">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;co-linear&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-belief-amp-ramp">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-dist">
      <value value="&quot;Uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-to-relocate">
      <value value="0"/>
      <value value="10"/>
      <value value="25"/>
      <value value="50"/>
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Voter11TippingFreq" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="200"/>
    <metric>fizz</metric>
    <metric>avgvote</metric>
    <metric>left-freq</metric>
    <metric>center-freq</metric>
    <metric>right-freq</metric>
    <enumeratedValueSet variable="global-belief-amp-ramp">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="local-belief-amp-ramp" first="0" step="0.025" last="0.3"/>
    <enumeratedValueSet variable="polluter-to-nonpolluter-ratio">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-entrench">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="init-attitude">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Influence">
      <value value="&quot;equal&quot;"/>
      <value value="&quot;co-linear&quot;"/>
      <value value="&quot;co-square&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnd-initial">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-fraction">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="relocation">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
