;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
globals
[
  ;; holds the actual value used to determine how many patches should choose
  ;; target-color.  this is set when the setup or rerun buttons are pressed.
  ;; if random-rerun? is true, this will be set to a random value from 0 to 100.
  ;; otherwise, this is set to whatever the current value of %-target-color is.
  real-%-target-color
  ;; Users guess the % of target-color in the population
  target-color  ;; set to green currently
  other-color  ;; set to blue currently
  hide-patch-color  ;; the color that patches are colored when they are not revealed
  ;; holds the number of patches in the observer's last sample
  num-patches-in-observer-sample
  ;;  holds the number of target color patches in the observer's last sample
  num-target-color-patches-in-observer-sample
  ;; if you were to segregate the patches by color, target-color to the left and
  ;; other-color to the right, you would create a contour line at approximately
  ;; xcor-of-organize-contour
  xcor-of-organize-contour
  max-$$  ;; the starting value for $$-remaining for the clients
  ;; variables for storing various statistics of guesses
  guesses-list  ;; list of all the clients guesses for this current population
  guess-average  ;; the average of all the students' guesses
  averages-list  ;; list of all the averages since we last pressed rerun or setup
  ;; various indicators of procedure status.
  revealing?  ;; true if population is revealed; false otherwise
  can-collect-more-data?  ;; true if we want to collect more data for this population
  have-committed?  ;; true if the commit button has been pressed before the plot guesses button was
  have-plotted?  ;; true if we have already called plotted data for this population
  ;; quick start instructions variables:
  quick-start   ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item       ;; the index of the current quickstart instruction
  qs-items      ;; the list of quickstart instructions
]
patches-own
[
  my-color  ;; variables for when user toggles between samples and reveal displays
  ;; a list of the users (including the observer/instructor/server and all active clients)
  ;; that have this patch in their current sample
  agents-with-me-in-sample
]
breed [ frames frame ]       ;; cause patches to have a thin frame surrounding them
breed [ gatherers gatherer ] ;; turtles that execute the special distributions that deviate from normal distribution
breed [ clients client ]     ;; each student is assigned a turtle that monitors the student's input
  ;; if a student logs in late, get assigned a pending client until rerun or
  ;; more data is pressed at which time it is converted into a client.
breed [ pending-clients pending-client ]
clients-own
[
  user-id  ;; unique id, input by the client when they log in, to identify each student turtle
  temp-my-guess  ;; the current value of the my-guess-for-this-sample slider on the hubnet client
  final-my-guess  ;; the value of temp-my-guess when the accumulate-data is called
  my-sample-block-side  ;; the current value of the my-sample-block-side slider on the hubnet client
  my-sampling-allowance  ;; the current number of patches that the client can still sample
  temp-go-with-group-guess?  ;; the current value of the go-with-group-guess? switch on the hubnet client
  final-go-with-group-guess?  ;; the value that temp-go-with-group-guess? had when the commit button was pressed
  ;; holds the amount of money the clients have.  this can be set to max-$$ by
  ;; pressing the replenish $$ button
  $$-remaining
]
pending-clients-own
[
  user-id  ;; unique id, input by the client when they log in, to identify each student turtle
]

;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;
to startup
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
  initialize true
  setup-quick-start
end
;; setup the model to be used
to initialize [ full-setup? ]
  no-display
  clear-plot
  initialize-variables
  initialize-patches
  hide-frames
  abnormality-distribution
  initialize-clients full-setup? ;; hubnet code
  if not allow-student-set-block-side?
    [ask clients
      [set my-sample-block-side sample-block-side]
     hubnet-broadcast "my-sample-block-side" sample-block-side
    ]
  display
end
;; initialize-variables when rerunning or setting up
to initialize-variables
  set revealing? false
  set max-$$ 100
  init-color-vars
  ;; the contour line between colors will be at a location proportionate to the
  ;; real-%-target-color of world-width.  for example, if real-%-target-color
  ;; is 70 then about 70% of the View will be solid target-color
  ;; beginning from the left and extending to the right.
  set xcor-of-organize-contour (min-pxcor) + (real-%-target-color * world-width / 100)
  set averages-list []
  start-over-variables
end
;; initialize the color-related variables
to init-color-vars
  set target-color 55
  set other-color 104
  set hide-patch-color 3
  ifelse random-rerun?
  [
    set real-%-target-color random 101
    set %-target-color 0
  ]
  [ set real-%-target-color %-target-color ]
end
;; initialize variables that are needed when collecting a set of data for the
;; current population
to start-over-variables
  set num-patches-in-observer-sample 0
  set num-target-color-patches-in-observer-sample 0
  set guesses-list []
  set guess-average 0
  set have-plotted? false
  set can-collect-more-data? false
  set have-committed? false
end
;; initialize patch variables
to initialize-patches
  set-default-shape frames "frame"
  ask patches
  [
    initialize-patch-colors
    set agents-with-me-in-sample []
    setup-frames
  ]
end
;; initialize patches colors.  this is where the patches determine whether they
;; are target-color or other-color
to initialize-patch-colors  ;; patch procedure
  set pcolor hide-patch-color  ;; default color of patches when not revealed
  ;; each patch "throws a die" to determine if it's target-color or other-color
  ifelse random-float 100.0 < real-%-target-color
  [ set my-color target-color ]
  [ set my-color other-color ]
end
;; each patch creates a frame which give the optional grid appearance
to setup-frames  ;; patch procedure
  if not any? frames-here
  [
    sprout-frames 1
    [
      ht
    ]
  ]
end
;; ask all the frames to hide, even if grid? is true
to hide-frames
  ask frames
  [ ht ]
end
;; create and kill gatherers which modify the bi-linear random distribution of
;; target-color and other-color patches in the population.  in other words, it
;; creates turtles which clump like-colored patches
to abnormality-distribution
  create-gatherers (4 + random 4)  ;; create gatherers
  [
    ht
    setxy random-xcor random-ycor
    gather
    die
  ]
end
;; clump like-colored patches by swapping patch colors from progressively
;; further patches around your location based on the abnormality slider
to gather  ;; gatherer procedure
  let index 0
  let a-patch 0
  let another-patch 0
  set index ( 20 * abnormality )
  while [ index > 0 ]
  [
    set a-patch one-of patches in-radius 16 with [ my-color = target-color ]
    set another-patch one-of patches in-radius 8 with [ my-color = other-color ]
    if (a-patch != nobody and another-patch != nobody)
    [ swap-my-color a-patch another-patch ]
    set index (index - 1)
  ]
end
;; swap my-colors between two patches
to swap-my-color [ a-patch another-patch ]
  let color-buffer 0
  set color-buffer [my-color] of a-patch
  ask a-patch [ set my-color [my-color] of another-patch ]
  ask another-patch [ set my-color color-buffer ]
end
;; setup the model for use.  this kills all the turtles and should only be used
;; with a new group
to setup
  if (user-yes-or-no? (word "Are you sure you want to setup? This will cause all "
                      "data to be lost.  If you just want to start another "
                      "round of guesses then press MORE DATA and if you want "
                      " a new population then press RERUN"))
  [
    ;; use no-display here so that we aren't sending the View
    ;; to the clients while doing ca
    no-display
    ca
    initialize true ;; initialize turns back on the display
    setup-quick-start
  ]
end
;; setup the model for use.  this doesn't kill all the turtles and so should be
;; used if you want to create a new population for the current set of clients.
to rerun
  if user-yes-or-no? "This will create a new population. Is this what you really want?"
  [ initialize false ]
end
;; setup the model for use.  this keeps the current population and set of
;; turtles, but sets up the model so that more data can be collected on the
;; current population.
to more-data
  if ( not any? clients ) or ( not can-collect-more-data? ) or ( not alert-no-rerun? )
  [ stop ]
  no-display
  start-over-patches
  set averages-list fput guess-average averages-list
  start-over-plot
  start-over-variables
  start-over-clients  ;; hubnet code
  ;; don't turn on the display if we are keeping the samples since
  ;; otherwise all the clients could see everybody else's samples
  if not keep-samples?
  [ display ]
end
;; make sure that the user really wants to use the same population
to-report alert-no-rerun?
  report user-yes-or-no? (word "You haven't pressed the RERUN button, so you'll have"
                          " the same population.  Is this ok with you?")
end
;; prepare the patches to be sampled with the same population
to start-over-patches
  ask patches
  [
    set pcolor hide-patch-color
    ;; don't clear the agents-with-me-in-sample list if we are keeping the samples
    if not keep-samples?
    [ set agents-with-me-in-sample [] ]
  ]
  hide-frames
end
;; plot means and mean of means
to start-over-plot
  let list-counter 0
  clear-plot
  set-current-plot-pen "means"
  set list-counter 0
  ;; plot a line for each item in averages-list
  repeat length averages-list
  [
    plot-pen-up
    plotxy (item list-counter averages-list)  0
    plot-pen-down
    plotxy (item list-counter averages-list) 25
    set list-counter list-counter + 1
  ]
  ;; if there are at least 2 items in averages-list, plot the mean of all the items
  if length averages-list > 1
  [
    set-current-plot-pen "mean-of-means"
    plot-pen-up
    plotxy mean averages-list 0
    plot-pen-down
    plotxy mean averages-list 25
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;
;; if the user clicks in the View, take a sample where the click
;; occurred.  the size of the sample depends upon the current value of the
;; sample-block-side slider
to sample
  if mouse-down?
  [
      ;; we use no-display here for efficiency.  we assume that if the
      ;; user is taking a sample then all clients should know about that
      ;; sample.  this is why we turn the display back on at the end.
      no-display
      change-sample (round mouse-xcor) (round mouse-ycor) sample-block-side "observer"
      display
  ]
end
;; based on input, create the chosen sample and have all patches in it set pcolor to my-color
to change-sample [ x y block-side agent ]
  let patches-in-sample 0
  set patches-in-sample nobody
  set revealing? false
  ;; designates all patches in a square around the mouse spot as being in the sample.
  ;; the sample-block side is set on the interface using the slider
  ask patch x y
  [ set patches-in-sample (in-rectangle patches ((block-side - 1) / 2)  ((block-side - 1) / 2) ) ]
  ;; hide all the patches and frames
  ask patches with [ true ]
  [ set pcolor hide-patch-color ]
  ask frames
  [ ht ]
  ;; show the patches their frames (if grid? is true) in the currently chosen sample
  ask patches-in-sample
  [
    set pcolor my-color
    ask frames-here
    [ set hidden? not grid? ]
  ]
  if keep-samples?
  [
    ;; show all the patches which were previously sampled while
    ;; keep-samples? was true
    ask patches with [ member? agent agents-with-me-in-sample ]
    [ set pcolor my-color ]
    ask patches-in-sample
    [
      ;; add agent to my list of agents that have me in their samples
      ;; so that the agent will keep seeing me when taking subsequent
      ;; samples (if keep-samples? is true)
      if( not member? agent agents-with-me-in-sample )
      [ set agents-with-me-in-sample fput agent agents-with-me-in-sample ]
    ]
  ]
  if agent = "observer"
  [
    set num-patches-in-observer-sample count patches-in-sample
    set num-target-color-patches-in-observer-sample count patches-in-sample with [pcolor = target-color]
  ]
end
;; report the agentset within a box that has a width = (width/2 * 2) + 1
;; and a height = (height/2 * 2) + 1 and is centered on the calling agent
to-report in-rectangle [ patchset width/2 height/2 ]  ;; turtle and patch procedure
  ;; this procedure does not work with a non-patch-agentset variable
  if not is-patch-set? patchset
  [ report nobody ]
  report patchset with
  [
    pxcor <= ([pxcor] of myself + width/2) and pxcor >= ([pxcor] of myself - width/2) and
    pycor <= ([pycor] of myself + height/2) and pycor >= ([pycor] of myself - height/2)
  ]
end
;; reveal the whole population - have all the patches show their 'true' colors
to reveal
  no-display
  set allow-student-sampling? false
  set num-patches-in-observer-sample 0
  set revealing? true
  if organize?
  [
    hubnet-broadcast "Message for you" (word "The percentage of the target color in the population is " real-%-target-color "%")
  ]
  ;; grid? is the option to have frames around each patch
  ask frames
  [ set hidden? not grid? ]
  ask patches
  [
    ifelse organize?
    [
      ;; When we reveal the population with organize? true, we get a vertical
      ;; contour. Patches with target-color are on the left and other
      ;; patches are on the right
      ifelse pxcor < xcor-of-organize-contour
      [ set pcolor target-color ]
      [ set pcolor other-color ]
     set %-target-color real-%-target-color
    ]
    [ set pcolor my-color ]
  ]
  display
end
;; reports the proportion of patches with target-color in the current sample
;; if there has been no sample chosen yet, patches-in-sample is nobody.  so
;; report -1 as an error code.
to-report target-color-%-in-sample
  ifelse (num-patches-in-observer-sample = 0)
  [ report -1 ]
  [ report ( 100 * num-target-color-patches-in-observer-sample / num-patches-in-observer-sample ) ]
end
;; plot the clients' guesses.  also, if commit has been called, update the
;; clients' $$-remaining
to plot-guesses
  if not continue?
  [ stop ]
  if not have-plotted?
  [ accumulate-data ]
  clear-client-messages  ;; hubnet code
  set-current-plot "Student Guesses"
  clear-plot
  set-current-plot-pen "guesses"
  set-histogram-num-bars 100 / cluster-guesses
  histogram guesses-list
  set-current-plot-pen "mean-of-guesses"
  plot-pen-up plotxy (mean guesses-list) plot-y-min
  plot-pen-down plotxy (mean guesses-list) plot-y-max
  set can-collect-more-data? true
  if have-committed? and not have-plotted?
  [ update-$$ ]
  set have-plotted? true
end
;; return true if a sufficient amount of students have submitted data.  the
;; teacher has an opportunity either to help students who haven't submitted
;; data or to just plow ahead.
to-report continue?
  let continuing? 0
  set continuing? true
  if have-plotted?
  [ report true ]
  if not any? clients
  [
    user-message "None of your students have connected.  You must collect data before continuing with this action."
    set continuing? false
  ]
  if any? clients with [ final-my-guess = -1 ]  ;; there is at least 1 student that hasn't submitted data
  [
    ifelse count clients = count clients with [ final-my-guess = -1 ]  ;; nobody has submitted data
    [
      user-message "No clients have submitted data."
      set continuing? false
    ]
    [
      ;; some (but not all) clients haven't submitted data
      if user-yes-or-no? (word "Please note that " (count clients with [ final-my-guess = -1 ])
          " students have not yet input new data.  Do you wish to include them in this round?")
      [ set continuing? false ]
    ]
  ]
  report continuing?
end
;; pools student data that you have collected from clients
to accumulate-data
  ;; get rid of -1 from the guesses-list since -1 is a default value and not a
  ;; valid guess
  set guesses-list remove -1 ([ final-my-guess ] of clients)
  set guess-average mean guesses-list
end
;; cause the final-go-with-group-guess? to be set for all the clients.
;; basically, make the clients commit to whether they are going with the group
;; guess
to commit
  ask clients
  [ set final-go-with-group-guess? temp-go-with-group-guess? ]
  if have-plotted? and not have-committed?
  [ update-$$ ]
  set have-committed? true
end
;; update $$-remaining for each of the clients' based on whether or not each
;; one chose to go with the group's average guess
to update-$$
  let group-error 0
  set group-error abs ( real-%-target-color - guess-average )
  ask clients
  [ update-client-$$ group-error ]
end
;; update the client's $$-remaining based on the value of final-go-with-group-guess?
to update-client-$$ [ group-error ]  ;; client procedure
  ifelse final-go-with-group-guess?
  [ update-$$-remaining group-error ]
  [ update-$$-remaining abs ( real-%-target-color - final-my-guess ) ]
  hubnet-send user-id "$$-remaining" $$-remaining
end
;; update the client's $$-remaining based on my-chosen-error and margin-of-error
to update-$$-remaining [ my-chosen-error ]  ;; client procedure
  if my-chosen-error > margin-of-error
  [ set $$-remaining round ( $$-remaining - abs ( my-chosen-error - margin-of-error ) ) ]
end
;; set $$-remaining of all clients to max-$$
to replenish-$$
  ask clients
  [ set $$-remaining max-$$ ]
  hubnet-broadcast "$$-remaining" max-$$
end
;; set my-sampling-allowance of all clients to total-sample-patches
to replenish-sampling-allowance
  ask clients
  [ set my-sampling-allowance total-sampling-allowance ]
  hubnet-broadcast "My sample allowance" total-sampling-allowance
  hubnet-broadcast "Message for you" "Your sampling allowance has been replenished!"
end
;; cause the View to show all the samples that students have
;; taken this far
to pool-samples
  ask patches
  [
    ifelse length agents-with-me-in-sample = 0
    [ set pcolor hide-patch-color ]
    [
      set pcolor my-color
      ;; put all the clients (including "observer") into agents-with-me-in-sample
      set agents-with-me-in-sample lput "observer" [ self ] of clients
    ]
  ]
  display
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients     ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; determines which client sent a command, and what the command was
to listen-clients
  ;; if a student can't set the block-size (before the first sample), then they
  ;; shouldn't be able to change it (between samples) either.
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-client ]
    [
      ifelse hubnet-exit-message?
      [ remove-client ]
      [ execute-cmd ]
    ]
  ]
end
;; if no clients with this user-id exist, create a new client and initialize its
;; variables.  if have-plotted? or have-committed? are true, create a
;; pending-client instead of a client
to create-client
  if not any? clients with [ user-id = hubnet-message-source ]
  [
    ifelse have-plotted?
    [
      create-pending-clients 1
      [ setup-client true hubnet-message-source ]
    ]
    [
      create-clients 1
      [ setup-client false hubnet-message-source ]
    ]
  ]
end
;; setup a client or a pending-client depending upon pending?
to setup-client [ pending? id ]  ;; client or pending-client procedure
  ht
  set user-id id
  ifelse pending?
  [
    hubnet-send user-id "Message for you"
                         (word "You are not included in this round.  You will be "
                         "included later.")
  ]
  [
    init-client-vars
    ifelse allow-student-set-block-side?
    [ set my-sample-block-side 1 ]
    [ set my-sample-block-side sample-block-side ]
    set $$-remaining max-$$
    hubnet-send user-id "my-sample-block-side" sample-block-side
    hubnet-send user-id "$$-remaining" $$-remaining
    hubnet-send user-id "My sample allowance" my-sampling-allowance
    hubnet-send user-id "Message for you" "Your target color is GREEN.  Guess percentage GREEN."
  ]
end
;; set some client variables to appropriate defaults
to init-client-vars  ;; client or pending-client procedure
  set my-sampling-allowance total-sampling-allowance
  set temp-go-with-group-guess? false
  set final-go-with-group-guess? false
  set temp-my-guess 0
  set final-my-guess -1
end
;; ask any? clients with this user-id to die
to remove-client
  ask clients with [ user-id = hubnet-message-source ]
  [ die ]
end
;; based on what hubnet-message-tag is, execute some code
to execute-cmd
  if hubnet-message-tag = "my-guess-for-this-sample"
  [
    ask clients with [ user-id = hubnet-message-source ]
    [ set temp-my-guess hubnet-message ]
    stop
  ]
  if hubnet-message-tag = "input-guess"
  [
    ask clients with [ user-id = hubnet-message-source ]
    [
      set final-my-guess temp-my-guess
      if not have-plotted?
      [
        hubnet-send hubnet-message-source "Message for you" (word "Thank you.  Your input is: " final-my-guess )
      ]
    ]
    stop
  ]
  if hubnet-message-tag = "my-sample-block-side"
  [
    ask clients with [ user-id = hubnet-message-source ]
    [
      ifelse allow-student-set-block-side?
      [ set my-sample-block-side hubnet-message ]
      [
          ;; currently, due to a bug in hubnet, we must make sure that we
          ;; ignore hubnet messages for sliders when the value being sent
          ;; by the client is equal to the client's current value
          if hubnet-message != sample-block-side
          [
            hubnet-send hubnet-message-source "my-sample-block-side" sample-block-side
            hubnet-send hubnet-message-source "Message for you" "Sorry, you can't change my-sample-block-side at this time"
          ]
      ]
   ]
   stop
  ]
  if hubnet-message-tag = "go-with-group-guess?"
  [
    ask clients with [ user-id = hubnet-message-source ]
    [ set temp-go-with-group-guess? hubnet-message ]
    stop
  ]
  if hubnet-message-tag = "View"
  [
    if not allow-student-sampling? [stop]
    no-display
    ask clients with [ user-id = hubnet-message-source ]
    [ client-sample ]
  ]
end
;; if clients are allowed to sample and this client's my-sampling-allowance is
;; large enough to take a sample of my-sample-block-side^2, have the client take
;; a sample of where at the location stored in hubnet-message up and update any
;; the appropriate variables.
;; Note: the display should be turned off when calling this procedure otherwise,
;; all connected clients will see the sample chosen by this client.
to client-sample  ;; client procedure
  let my-total 0
  let patch-clicked-on 0
  if not allow-student-sampling? [stop]
  hubnet-send hubnet-message-source "Message for you" ""
  if not allow-student-set-block-side? and my-sample-block-side != sample-block-side
  [
    set my-sample-block-side sample-block-side
    hubnet-send hubnet-message-source "my-sample-block-side" sample-block-side
    hubnet-send hubnet-message-source "Message for you" (word "Sorry, my-sample-block-side must be " sample-block-side " now.")
  ]
  set patch-clicked-on patch (round item 0 hubnet-message) (round item 1 hubnet-message)
  setxy [pxcor] of patch-clicked-on [pycor] of patch-clicked-on
  set my-total my-sampling-allowance - count
      ( in-rectangle patches ((my-sample-block-side - 1) / 2) ((my-sample-block-side - 1) / 2) )
  ifelse my-total < 0
  [ hubnet-send hubnet-message-source "Message for you" "Sorry, that's more than your allowance." ]
  [
    change-sample pxcor pycor my-sample-block-side self
    set my-sampling-allowance my-total
    hubnet-send hubnet-message-source "View" "View"
    hubnet-send hubnet-message-source "My sample allowance" my-total
  ]
end
;; setup the clients so that they are ready to sample a new population
to initialize-clients [ full-setup? ]
  start-over-clients
  if full-setup?
  [ replenish-$$ ]
end
;; setup the clients for collecting another set of data for the current population
to start-over-clients
  convert-to-clients
  ask clients
  [ init-client-vars ]
  hubnet-broadcast "go-with-group-guess?" false
  hubnet-broadcast "My sample allowance" total-sampling-allowance
  hubnet-broadcast "my-guess-for-this-sample" 0
  hubnet-broadcast "Message for you" "Your target color is GREEN.  Guess percentage GREEN."
end
;; have all pending-clients become a client
to convert-to-clients
  ask pending-clients
  [
    set breed clients
    setup-client false user-id  ;; use false here since we have a real client now
  ]
end
;; erases any announcements which we previously sent to clients
to clear-client-messages
  ask clients
  [ hubnet-send user-id "Message for you" "" ]
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "SETUP initializes all variables but it also disconnects the users. You needn't press it now..."
    "use it only if you wish to begin the simulation again."
    "Normally, choose RERUN to initialize variables."
    "If you've just begun, note that the slider over the View is set to 50% target color,"
    "and the RANDOM-RERUN? switch is set to Off and the ABNORMALITY slider is set to 0."
    "Note: the target color is green and the other color is blue."
    "Now press REVEAL POP. to see the current population:" "50% of the patches are green; you have 50% greenness."
    "Practice setting the population to a different percentage and distribution type by..."
    "setting the percentage green for the population by..." "setting the %-TARGET-COLOR slider over the View."
    "ABNORMALITY controls how much the distribution of green deviates from 'normal'. Explore it."
    "If RANDOM-RERUN? is On then the computer will choose the percentage green for you."
    "That means that even you will not know the percentage green in the population."
    "Press the RERUN button for a new population."
    "REVEAL POP. varies according to the settings of ORGANIZE? and GRID?."
    "ORGANIZE? Off shows the colors where they are and organize On does color separation"
    "(see how the green/blue contour falls exactly under the %-TARGET-COLOR slider handle)"
    "GRID? On puts frames around the patches and GRID? Off does not."
    "SAMPLE allows you to reveal by clicking with the mouse square areas in the population."
    "The size of these square samples depends on the value of the SAMPLE-BLOCK-SIDE slider."
    "Set KEEP-SAMPLE? to choose whether or not to keep successive samples displayed."
    "In your own preparation for class, in order to anticipate students' questions,"
    "you should practice by using RANDOM-RERUN? and taking samples."
    "Think the way your students would. There are many questions to be asked..."
    "and especially when you play with ABNORMALITY and get color clusters in your population."
    "Work on strategies for maximizing the accuracy AND efficiency of your sample measurement:"
    "Should you take small samples or big ones?; just one sample or many?; where from?"
    "To start off your students, do more or less what you did yourself:"
    "Set the sliders and switches as following:"
    "%-TARGET-COLOR 50; ABNORMALITY 0; RANDOM-RERUN?, ORGANIZE?, and GRID? Off."
    "Press REVEAL POP."
    "Now ask students 'What is this?' 'How green is this?', 'How could we figure out?'"
    "To show the class how green it is, set ORGANIZE? to On and press REVEAL POP."
    "Draw students' attention to the contour line..." "and how it compares to their guess on the slider above it:"
    "Did they over/under-guess?; by how much? How well did they do? Were they close enough?"
    "Try this with other %. Now you'll want students to open the HubNet client, login."
    "When they login, each should see their own SAMPLER interface."
    "Now they can input their own guesses as long as COLLECT DATA is pressed."
    "COLLECT DATA imports values from all active participants -- always keep this pressed down."
    "Ask the students to set their sliders to their guesses and press their INPUT-GUESS buttons."
    "PLOT GUESSES calculates the statistics of participants' data and..." "displays them in the plot and monitors."
    "The logic of the $$ game is that students bet on either their own guess or the group guess;"
    "Press COMMIT to register each student's decision whether or not to go with the group guess."
    "Students each 'pay' for their error:" "As many points are deducted from their $$ as their guess was off."
    "But you can set the MARGIN-OF-ERROR slider to allow for more flexibility."
    "When the basic procedure is clear to students, switch On the ALLOW-STUDENT-SAMPLING?."
    "Students can sample as many patches as the sampling allowance is set to."
    "Press REPLENISH SAMPLING ALLOWANCE if you want to..." "reset all students to TOTAL-SAMPLING-ALLOWANCE."
    "Discuss individual and collaborative sampling strategies and..." "their relation to the histogram distribution."
    "Good luck!"
   ]
   set quick-start (item qs-item qs-items)
end
;; view the next item in the quickstart monitor
to view-next-message
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end
;; view the previous item in the quickstart monitor
to view-previous-message
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end

to turtle-draw
  if mouse-down?     ;; reports true or false to indicate whether mouse button is down
    [
     ask turtles-on patch mouse-xcor mouse-ycor
       [
;        ask turtles in-radius brush-size [set color draw-color]
         if clients-see
            [
             ask patches in-radius brush-size [set pcolor draw-color]
            ]
         ask patches in-radius brush-size [ set my-color draw-color]
        ]
    ]
end
to patches-draw
  ;if mouse-down?     ;; reports true or false to indicate whether mouse button is down
  ;  [
  ;   ask patches-at mouse-xcor mouse-ycor;;
;        [ask patches in-radius brush-size [set pcolor draw-color set mycolor draw-color]]
 ;   ]
end


to erase-all
ask turtles [if pcolor != white [set color red set heading color]]
end
@#$#@#$#@
GRAPHICS-WINDOW
27
43
403
440
30
30
6.0
1
10
1
1
1
0
1
1
1
-30
30
-30
30
0
0
1
ticks
CC-WINDOW
5
629
976
724
Command Center
0
BUTTON
613
362
728
395
Plot Guesses
plot-guesses
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
530
233
671
266
sample-block-side
sample-block-side
1
11
3
2
1
NIL
HORIZONTAL
PLOT
9
440
404
568
Student Guesses
 %-target-color
# students
0.0
100.0
0.0
6.0
true
true
PENS
"guesses" 1.0 1 -2674135 true
"mean-of-guesses" 1.0 0 -5825686 true
"means" 1.0 0 -6459832 true
"mean-of-means" 1.0 0 -11221820 true
BUTTON
420
189
528
222
Reveal Pop.
reveal
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SWITCH
531
190
670
223
organize?
organize?
0
1
-1000
SWITCH
422
359
542
392
grid?
grid?
0
1
-1000
MONITOR
80
570
163
615
# guesses
length guesses-list
0
1
11
MONITOR
164
570
268
615
mean this round
guess-average
1
1
11
MONITOR
440
570
539
615
mean all rounds
mean averages-list
1
1
11
MONITOR
366
570
439
615
# rounds
length averages-list
0
1
11
MONITOR
269
570
365
615
standard dev
standard-deviation guesses-list
3
1
11
MONITOR
5
570
79
615
# students
count clients
0
1
11
MONITOR
416
11
966
56
Quick Start Instructions (more details in the Information tab and part-sims guide)
quick-start
3
1
11
BUTTON
844
63
966
96
NEXT >>>
view-next-message
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
576
63
697
96
<<< PREV
view-previous-message
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
776
269
966
302
margin-of-error
margin-of-error
0
10
0
1
1
NIL
HORIZONTAL
SLIDER
26
10
408
43
%-target-color
%-target-color
0
100
35
1
1
NIL
HORIZONTAL
BUTTON
418
112
525
145
Rerun
rerun
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
422
403
542
436
cluster-guesses
cluster-guesses
1
10
1
1
1
NIL
HORIZONTAL
BUTTON
420
233
528
266
Sample
sample
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
SWITCH
530
269
671
302
keep-samples?
keep-samples?
0
1
-1000
BUTTON
670
396
787
429
More Data
more-data
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
670
328
787
361
Collect Data
listen-clients
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
416
64
544
97
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
528
146
670
179
abnormality
abnormality
0
10
0
1
1
NIL
HORIZONTAL
BUTTON
700
63
842
96
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
833
305
965
339
Replenish $$
replenish-$$
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SWITCH
770
201
966
234
allow-student-sampling?
allow-student-sampling?
1
1
-1000
BUTTON
419
310
527
343
Pool Samples
pool-samples
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
771
111
965
144
total-sampling-allowance
total-sampling-allowance
0
500
500
25
1
NIL
HORIZONTAL
BUTTON
729
362
845
395
Commit
commit
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
771
146
965
180
Replenish Sampling Allowance
replenish-sampling-allowance
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SWITCH
528
112
671
145
random-rerun?
random-rerun?
1
1
-1000
MONITOR
859
447
967
492
class-mean-$$
mean [$$-remaining] of clients with [$$-remaining > -1]
3
1
11
SWITCH
731
235
966
268
allow-student-set-block-side?
allow-student-set-block-side?
0
1
-1000
SLIDER
426
531
518
564
brush-size
brush-size
0
30
10
1
1
NIL
HORIZONTAL
CHOOSER
426
483
518
528
draw-color
draw-color
55 104
0
BUTTON
425
446
519
479
NIL
turtle-draw
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
MONITOR
522
483
600
528
my-color
count patches with [my-color = target-color]
3
1
11
SWITCH
693
468
810
501
clients-see
clients-see
0
1
-1000
BUTTON
522
446
600
479
hide-drawing
ask patches [set pcolor hide-patch-color]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
@#$#@#$#@
VERSION
-------
$Id: Sampler Monte Carlo.nlogo 39722 2008-05-01 18:38:33Z everreau $

WHAT IS IT?
-----------
Sampler is a HubNet Participatory Simulation in statistics. It is part of the ProbLab curricular models. Students engage in statistical analysis as individuals and as a classroom. Through these activities, students discover the meaning and use of basic concepts in statistics. Students take samples from a hidden population and experience the mathematics of statistics, such as mean, distribution, margin of error, etc. The graphics in the SAMPLER interface are designed to ground students' understanding of statistics in proportional judgments of color distribution. The collaborative tools are designed to help students appreciate the power of large numbers for making inferences about populations. Students experience distributions both at an individual level -- variation in their own samples -- and at a group level -- variation in all students' guesses. This analogy is designed for students' to appreciate the diversity of opinions in the classroom and the power of embracing everyone to achieve a complex task.

LEARNING STATISTICS
-------------------
In SAMPLER, statistics is presented as a task of making inferences about a population under conditions of uncertainty and limited resources. For example, if you wanted to know what percentage of students in your city speak a language other than English, how would you go about it? Would it be enough to measure the distribution of this variable in your own class? If yes, then how sure could you be that your statistic is representative of the whole city? If not, why not? Are there certain groups of people that it would make more sense to use as a sample? Are there other groups it would make no sense to use? For instance, would it make sense to stand outside a movie house that is showing a French film with no subtitles and ask each patron whether they speak a second language? Is this a representative sample? Should we look at certain parts of town? Would all parts of town be the same? Oh, and by the way, what is an average (a mean)? A variable? A value? What does it mean to measure a distribution of a variable within a population?
Many students have a very difficult time understanding statistics--not only in middle and high school, but also in college and beyond. Yet on the other hand, there are certain visual-mental capabilities we all have--even very young children--that could be thought of as naive statistics. These capabilities are the proportional judgments we make constantly. We make proportional judgments when we need to decide how to maximize the utility of our actions. For instance, when we come to a new place we may say, "People in this town are very nice." How did we decide that? Or, "Don't buy fruit there--it's often overripe." How did we infer that? Or, "To get to school, take Main street--it's the fastest route in the morning; but drive back through High street, I find that's faster in the afternoon."

HOW DOES IT WORK?
-----------------
The teacher works in NetLogo and acts as the server for the students (the "clients") who each have their own client interface on their computer screens. Students see the teacher's interface projected on the classroom screen, and they can instruct the teacher to manipulate settings of the microworld that they do not have on their own client interfaces. The View in the projected interface features a square "population" of 3721 squares. Individual patches are either green (target-color) or blue (other-color). The squares' color is the attribute we measure in SAMPLER. So, the SAMPLER color is a variable that can have one of two values: green or blue (a dichotomous variable, like a coin). In a basic SAMPLER activity, students and/or the teacher reveal(s) parts of or all the population and students discuss, approximate, take samples, and input their individual guesses as to the percentage of green patches within the revealed sector of the population. All participating students' inputs are collected, pooled, and represented in monitors and in the plot. Thus, each student constitutes a data-point agent and can experience impacting the class statistics.
Through collaboration, students are to achieve, as a class, the best possible approximation of the population.
The $$ game: At the beginning of every round and later, whenever the facilitator decides, all clients receive max-points, for instance $100. Now, students can bet either on their own guess or on the group guess. They pay 1 point for every percentile their bet is away from the truth or from the margin of error that they agree upon.  This is an optional feature.

HOW TO USE IT
-------------
QUICKSTART INSTRUCTIONS:
------------------------
SETUP initializes all variables but it also disconnects the users. You needn't press it now; use it only if you wish to begin the simulation again. Normally, choose RERUN to initialize variables for using Sampler with the same class. If you've just begun, see that the slider over the View is set to 50% target color, and the RANDOM-RERUN? switch is set to Off and the ABNORMALITY slider is set to 0. Note: the target color is green and the other color is blue.
Now press REVEAL POP. to see the current population: 50% of the patches are green; you have 50% greenness. Practice setting the population to a different percentage and distribution type by setting the percentage green for the population by setting the %-TARGET-COLOR slider over the View. ABNORMALITY controls how much the distribution of green deviates from 'normal'. Explore it.
If RANDOM-RERUN? is On then the computer will choose the percentage green for you. That means that even you will not know the percentage green in population. Press the RERUN button for a new population. REVEAL POP. varies according to the values of ORGANIZE? and GRID?. ORGANIZE? Off shows the colors where they are and ORGANIZE? On does color separation (see how the green/blue contour falls exactly under the %-TARGET-COLOR slider handle). GRID? On puts frames around the patches and GRID? Off does not. SAMPLE allows you to reveal with the mouse square areas on the View. The size of these square samples depends on the value of the SAMPLE-BLOCK-SIDE slider. Set KEEP-SAMPLE? to choose whether or not to keep successive samples displayed.
In your own preparation for class, in order to anticipate students' questions, you should practice by using RANDOM-RERUN? and taking samples. Think the way your students would. There are many questions to be asked and especially when you play with ABNORMALITY and get color clusters in your population. Work on strategies for maximizing the accuracy AND efficiency of your sample measurement: Should you take small samples or big ones?; just one sample or many?; where from?
To start off your students, do more or less what you did yourself:
-Set the sliders and switches as following: %-TARGET-COLOR 50; ABNORMALITY 0; RANDOM-RERUN?, ORGANIZE?, and GRID? Off.
-Press REVEAL POP.
-Now ask students 'What is this?' 'How green is this?', 'How could we figure out?'
-To show the class how green it is, set ORGANIZE? to On and press REVEAL POP.
-Draw students' attention to the contour line and how it compares to their guess on the slider above it: Did they over/under-guess?; by how much? How well did they do? Were they close enough?
-Try this with other %.
Now you'll want students to open the HubNet client and login. When they login, they should each see their own SAMPLER interface. Now they can input their own guesses as long as COLLECT DATA is pressed. COLLECT DATA imports values from all active participants -- always keep this pressed down. Ask the students to set their sliders to their guesses and press their INPUT-GUESS buttons. PLOT GUESSES calculates the statistics of participants' data and displays them in the plot and monitors.
The logic of the $$ game is that students bet on either their own guess or the group guess; press COMMIT to register each student's decision whether or not to go with the group guess. Students each 'pay' for their error: As many points are deducted from their $$ as their guess was off. But you can set the MARGIN-OF-ERROR slider to allow for more flexibility.
When the basic procedure is clear to students, switch On the ALLOW-STUDENT-SAMPLING?. Students can sample as many patches as the sampling allowance is set to. Press REPLENISH SAMPLING ALLOWANCE if you want to reset all students to TOTAL-SAMPLING-ALLOWANCE. Discuss individual and collaborative sampling strategies and their relation to the histogram distribution. Good luck!
See the SAMPLER section of the Computer HubNet Participatory Simulations Guide for an in-depth lesson plan.
BUTTONS:
--------
SETUP - clears all turtles and patches and the plot. This button should only be pressed when starting out with a new group of users since all data is lost.
RERUN - creates a new population from which to sample. In creating this population, each patch has either a random (if RANDOM-RERUN? is true) or a user chosen (if RANDOM-RERUN? is false) percent chance of being green. The user chosen chance is set with the %-TARGET-COLOR slider. This button should be used to setup the model again for collecting data with a new population and the same users connected.
SAMPLE - allows the server to reveal mouse-selected square areas of SAMPLE-BLOCK-SIDE within the population. Set the KEEP-SAMPLES? switch to choose whether or not to keep successive samples.
COLLECT DATA - collects student samples and guess values from all active clients. Clients, if allowed to sample, can reveal as many patches as they have left in their sample allowance. One sample allowance unit allows you to reveal one patch.
POOL SAMPLES - shows all samples (from the clients and the server) at once
REVEAL POP. - shows all the patches' colors. If ORGANIZE? is false, the patches reveal whatever color they got when RERUN was pressed. If ORGANIZE? is true, the patches segregate themselves by color, green to the left and blue to the right. If GRID? is true, each of patches have a thin frame surrounding it.
PLOT GUESSES - histograms the collected guesses in the plot. Once you have plotted, guesses are no longer accepted for this population until RERUN or MORE DATA are pressed.
MORE DATA - prepares the model for another round of class guesses for this specific population. To create a new population, use the RERUN button.
REPLENISH SAMPLING ALLOWANCE - resets each of the client's sampling allowance to TOTAL-SAMPLE-PATCHES
COMMIT - collects students' current setting of their GO-WITH-GROUP? switch.  This is used for the $$ game, and is not needed if you are not doing the game.
REPLENISH $$ - resets each of the client's $$-REMAINING to the starting quantity.  Note: clients' $$-REMAINING are never replenished unless you press this button.  We suggest that a good time to press it might be when you press the RERUN button.
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction
SLIDERS:
--------
ABNORMALITY - slider controls to what extent the distribution deviates from 'normal' (for a given percent green you'll get larger clumps for a larger setting)
MARGIN-OF-ERROR - used for the $$ game. This determines how accurate the guess has to be in order to be correct. For example, if it's set at 3 and the greenness is 70 then you can guess between 67 and 73 and not have points taken off, but if you guess 74 or 66 you get 1 point off, etc.
TOTAL-SAMPLE-PATCHES - determines how much sampling allowance students get when it is replenished. When sampling, one sample allowance unit allows you to reveal one patch.
SAMPLE-BLOCK-SIDE - determines the side of the square that is revealed when sampling in the model. For instance, SAMPLE-BLOCK-SIDE of 5 gives a sample size of 25 patches. This value does not affect how big the samples are on the clients. They have their own versions of this slider.
CLUSTER-GUESSES - controls the histogram interval: the higher the setting, the higher the interval.
SWITCHES
--------
ALLOW-STUDENT-SAMPLING? - if true, students can sample; otherwise not.
ALLOW-STUDENT-SET-BLOCK-SIDE? - if true, students can only set the size of their samples before sampling for the first time that round; otherwise not.
ORGANIZE? - if true, when REVEAL is pressed, the green and blue colors will be segregated (green on the left).  If false, then when you press REVEAL, the colors will be not be organized.
GRID? - if true, each patch will have a thin frame around it to help you count them.  If false, then the frames will not be seen.
RANDOM-RERUN? - when RERUN is pressed, each patch has either a random (if RANDOM-RERUN? is true) or a user chosen (if RANDOM-RERUN? is false) percent chance of being green. The user chosen chance is set with the %-TARGET-COLOR slider. This button should be used to setup the model again for collecting data with a new population and the same users connected.
KEEP-SAMPLES? - when sampling, if true, old samples are still displayed.  If false, old samples are removed and cannot be seen.
MONITORS
--------
SAMPLE SIZE - shows the number of patches in the current sample chosen in the NetLogo model (the teacher's screen)
CLASS MEAN $$ - shows the mean of students' $$-remaining but only students whose $$-remaining is 0 and above
SAMPLE/POP - shows the quotient of TARGET COLOR % IN THIS SAMPLE and TARGET COLOR % IN POPULATION. So, in a sense, it tells you how indicative a sample is of the population statistic. If the quotient is exactly 1 then the sample can be said to be representative of the population. If the quotient is smaller than 1 then the sample under represents the population, and if the quotient is larger than 1 then the sample over represents the population statistic. Larger samples generally give quotients that are closer to 1.  Only when SHOW-SAMPLE-%? and SHOW-POP-%? are true does this monitor show any pertinent information.
# STUDENTS - shows how many students are actively connected to the NetLogo model (the teacher's screen)
# GUESSES - shows how many guesses were collected when you last pressed PLOT GUESSES.
MEAN THIS ROUND - shows the average of guesses that are currently plotted in the histogram.
STANDARD DEV - shows the standard deviation of guesses
# ROUNDS - shows how many rounds are represented in the plot.
MEAN ALL ROUNDS - the cumulative average for all rounds per this population (since you last pressed RERUN).
PLOTS
-----
AVERAGES OF STUDENT GUESSES- X-axis is %-TARGET-COLOR and Y-axis is # STUDENTS. Here you see four statistics as displayed by four different plot pens:
1. GUESSES: Students' collected guesses for a round represented in histograms.
2. MEAN-OF-GUESSES: the average value of guesses for the recent round
3. MEANS: the average values from successive rounds
4. MEAN-OF-MEANS: the average value of 'means'.

CLIENT INFORMATION
------------------
MY-GUESS-FOR-THIS-SAMPLE - students use this slider to set their guess value.
MY-SAMPLE-BLOCK-SIDE - students use this slider to set the size of their sample square
GO-WITH-GROUP-GUESS? - students use this switch to set whether or not they are committing themselves to go with the class average guess.
MESSAGE FOR YOU - displays messages sent from the server; these may be collective or for subsets of students.
MY SAMPLE ALLOWANCE - shows how many patches the student may still sample.
$$-REMAINING - shows how many dollar points the student still has left.
INPUT-GUESS - press this button to send to the server the guess that you have set on the MY-GUESS-FOR-THIS-SAMPLE slider.

THINGS TO NOTICE
----------------
When you press REVEAL in the organize-on option, the target-color and the other-color move to the left and the right of the screen, respectively, forming a contour line. The location of this contour line is comparable to two other elements on the interface: the contour line falls directly below the slider handle above it (if this was not a random run) and it relates similarly to the mean line in the plot. The reason we can compare these three features directly is because the 0 and 'whole' (100%) of each of these features are aligned. That is, the sliders, View, and plot have all been placed carefully so as to subtend each other precisely.
The abnormality distribution feature does not take much code to write, but is effective. Look at the code and try to understand it.

THINGS TO TRY
-------------
Set RANDOM-RERUN? to true and press RERUN and take samples. What is the minimal number of samples you need in order to get a good idea of the target-color distribution in the population? Anyway, how 'good' must a good idea be? Can you think of a way of describing this 'goodness'? What is a good way of spreading the samples on the population?
Try setting the ABNORMALITY slider to different values and press RERUN over and over for the same percentage green, for instance 50%. Can you think of situations in the world where a certain attribute is distributed in a population in a way that corresponds to a high value of ABNORMALITY? What do we mean when we speak of a 'uniform distribution' within a population? For instance, is a distribution of ABNORMALITY = 0 uniform? Or must there be strict order, for instance stripes of target-color, in order for you to feel that the distribution is uniform? Also, is there a difference between your sense of uniformity whether you're looking at the whole population or just at certain parts of it? If you threw a handful of pebbles onto a square area, would you say they fell 'uniformly'? What kinds of patterns are natural, and what kinds of patterns would you think of as coincidental?
Set RANDOM-RERUN? to false and press RERUN multiple times.  Each time you press RERUN, change the SHOW-POP-%? to true.  Do you notice that the number of green patches varies from trial to trial?  The way the population is established every trial is that each patch "flips a weighted coin" to see whether it should be green or not. Because there are so many patches, it turns out that the percentage of patches that "landed on" green is roughly the same as the percentage green you set with the slider. You can explore this intriguing idea further in the ProbLab model Stochastic Patchwork.

EXTENDING THE MODEL
-------------------
What other quantitative aspects of sampling might a teacher or student need so as to understand and do more in this activity? Perhaps the class would want to keep a record of how well they are doing over an entire lesson. How would you quantify such performance and how would you display it? Would a plot be useful for this or just a list of numbers?

RELATED MODELS
--------------
All models in ProbLab deal with probability and statistics in ways that may enrich student understanding of sample space, randomness, and distributions. In particular, many models share with SAMPLER the 3-by-3 sample that we call a "9-block."

CREDITS AND REFERENCES
----------------------
Thanks to Dor Abrahamson for designing this and other models in ProbLab.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250
airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15
arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150
box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75
bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30
butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60
car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58
circle
false
0
Circle -7500403 true true 0 0 300
circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123
face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240
face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225
face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183
fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30
flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45
flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240
frame
false
14
Rectangle -7500403 false false 0 1 296 296
Rectangle -7500403 false false 0 2 297 297
Rectangle -7500403 false false 0 0 298 298
Rectangle -7500403 false false 1 3 295 295
Rectangle -7500403 false false 1 1 299 299
house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120
leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195
line
true
0
Line -7500403 true 150 0 150 300
pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120
person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
square
false
0
Rectangle -7500403 true true 30 30 270 270
square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240
star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108
target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60
tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152
triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255
triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224
truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42
turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99
wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269
x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 4.1pre1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
40
10
406
376
30
30
6.0
1
10
1
1
1
0
1
1
1
-30
30
-30
30
BUTTON
167
445
267
478
input-guess
NIL
NIL
1
T
OBSERVER
NIL
NIL
SLIDER
36
410
417
443
my-guess-for-this-sample
my-guess-for-this-sample
0
100
0
1
1
%
HORIZONTAL
MONITOR
12
479
439
528
Message for you
NIL
3
1
SLIDER
12
530
197
563
my-sample-block-side
my-sample-block-side
1
11
3
2
1
NIL
HORIZONTAL
MONITOR
352
530
441
579
$$-remaining
NIL
0
1
SWITCH
12
565
197
598
go-with-group-guess?
go-with-group-guess?
1
1
-1000
MONITOR
207
530
338
579
My sample allowance
NIL
3
1
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
