;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  clock             ;; keeps track of the number of times modulus total-intervals through the go procedure
  num-intervals     ;; the number of time intervals a elevator can move
  received-info?    ;; equals true if data has been sent from the calculators

  ;; variables for leaving a trail
  up-trail-color    ;; color of the trail left behind by an elevator as it moves upwards
  down-trail-color  ;; color of the trail left behind by an elevator as it moves downwards

  ;; plotting variables
  users             ;; list of the user ids for the current set of elevators
  elevator-to-plot  ;; the elevator to plot if plot-all-elevators? is false
  old-plot-all-elevators?  ;; the value of plot-all-elevators? in the last pass thru the go procedure

  ;; quick start instructions variables
  quick-start       ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item           ;; the index of the current quickstart instruction
  qs-items          ;; the list of quickstart instructions
]

turtles-own
[
  ;; variable used in interactions with calculators
  user-id           ;; the id of a turtle which corresponds to the user-id of a calculator

  ;; position variables
  my-ycor           ;; the unwrapped vertical position of a turtle
  ycor-initial      ;; the initial vertical position of a turtle

  ;; variable to keep track of the time
  interval          ;; the current interval a turtle is moving through

  ;; variables keeping the velocities of a turtle
  velocity          ;; current velocity of an elevator
  velocities        ;; list of velocities
]

patches-own
[
  base-pcolor       ;; the original color of the patch before any trail was drawn on it
]

;;;;;;;;;;;;;;;;;;;;;;
;; Setup Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;

to startup
  setup
  setup-quick-start
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
end

;; Initializes the display, the plots, and the global variables.
to setup
  setup-patches
  setup-vars
  setup-plots
  set-default-shape turtles "elevator"
end

;; setup the model to be ready to be run with
;; the current elevators and with clock set to zero
to re-run
  reset-elevators
  setup-plots
end

;; set up the patches to have the floor numbers down the left side of the
;; view and color all even floor a gray color
to setup-patches
  clear-output
  cp
  ct

  ;; give the floors some color to be able to distinguish one floor from another
  ask patches with [ pycor mod 2 = 0 ]
  [ set pcolor 3 ]
  ask patches with [ pycor = 0 ]
  [ set pcolor 4 ]
  ;; label each row of pycors with a floor number
  ask patches with [ pxcor = min-pxcor ]
  [ set plabel pycor ]
  ask patches
  [ set base-pcolor pcolor ]
end

;; set variables to initial values
to setup-vars
  set clock 0
  set num-intervals 9
  set received-info? false
  set up-trail-color green
  set down-trail-color red

  ;; by default have the elevator to plot be nobody
  set elevator-to-plot nobody
end

;; give the user some information about what the setup button does so they can
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? (word "The SETUP button should only be used when starting "
              "over with a new group (such as a new set of students) since "
              "all data is lost.  Use the RE-RUN button for continuing with "
              "an existing group."
              "\\n\\nDo you really want to setup the model?")
  [
    user-message (word "Before closing this dialog, please do the following:"
                 "\\n  -Have everyone that is currently logged in, log off and "
                 "clear the calculator memory. (Press 2nd MEM 7 1 2)"
                 "\\n  -Open the teacher console for this activity and press the ERASE ALL DATA button.")
    setup
  ]
end

;; creates a set of num-turtles elevators.  only used in testing.
to make-test-turtles [ num-turtles ]
  crt num-turtles
  [
    set ycor-initial 3
    set color 5 + (10 * random 14)
    set heading 0
    set interval 0
    set user-id (word who)
    ifelse (who mod 3 = 0)
    [ set velocities [ 1 2 3 2 1 0 -1 0 0 ] ]
    [
      ifelse (who mod 3 = 1)
      [ set velocities [ 0 0 1 -1 3 -2 -1 0 0 ] ]
      [ set velocities [ 0 0 -1 4 3 -2 -3 0 5 ] ]
    ]

    ;; set initial positions of the elevators
    set my-ycor ycor-initial
    set ycor ycor-initial

    ifelse (abs my-ycor <= max-pycor)
    [ set hidden? false ]
    [ set hidden? true ]
    set label-color yellow
    show-or-hide-id-labels
  ]

  set received-info? true
end


;;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;

;; move the elevators one interval
to go
  listen-clients
  every delay  ;; we want to move to a new interval after delay seconds
  [
    if clock < num-intervals
    [
      ask turtles
      [
        show-or-hide-id-labels  ;; shows or hide the turtles labels
        assign-values           ;; set the turtles' velocity to the appropriate value
        move-elevators          ;; move the elevators by their velocity for this interval
      ]
      do-plotting
      ask patches with [ pcolor != base-pcolor ]
        [ fade-trail ]
      set clock clock + 1
    ]
  ]
end

;; return the elevators to their initial positions and reset the clock
to reset-elevators
  set clock 0
  ask patches [ set pcolor base-pcolor ]
  ask turtles
  [
    set heading 0
    set ycor ycor-initial
    set interval 0
    set my-ycor ycor-initial
    st
    show-or-hide-id-labels
  ]
end

;; calculate the delay in seconds based on the simulation speed
to-report delay
  ifelse simulation-speed <= 0
  [ report ln (10 / 0.001) ]
  [ report ln (10 / simulation-speed) ]
end

to show-or-hide-id-labels  ;; turtle procedure
  ifelse show-user-id?
  [ set label word user-id "     " ]
  [ set label "" ]
end

;; set the student selected elevator velocity for the current interval
to assign-values  ;; turtle procedure
  ifelse interval >= length velocities
  [ set velocity 0 ]
  [ set velocity item interval velocities ]
end

;; move each turtle by its velocity for this interval
to move-elevators  ;; turtle procedure
  let delta-pos 0

  if velocity != 0
  [ set delta-pos (velocity / abs velocity) ]
  let inner-tick 0

  while [ inner-tick < abs velocity ]
  [
    ;; divide the amount of time till the next interval into equal amounts
    ;; so as to be able to make the motion of an elevator smooth
    every ( delay / (abs velocity) )
    [
      ifelse (velocity > 0)  ;; leave a trail behind the turtle.  the color of the trail will depend upon whether the turtle is moving up or down
      [ set pcolor up-trail-color ]
      [ set pcolor down-trail-color ]

      set my-ycor (my-ycor + delta-pos)  ;; add to the turtle's unwrapped ycor 1 or -1 based on the sign of its velocity
      ifelse patch-ahead delta-pos != nobody
      [ st
        fd delta-pos ]  ;; if my-ycor is on the screen, then have the turtle color the patch that it is on
      [ ht ]  ;; otherwise, hide this turtle so it doesn't wrap around the screen, but appears to go off the edge of the screen
      set inner-tick (inner-tick + 1)  ;; increase the inner clock of the turtle
    ]
  ]

  set interval (interval + 1)
end

;; have any trails fade back to the base color of the patch
to fade-trail  ;; patch procedure
  set pcolor pcolor - 0.4
  if (pcolor mod 10 = 0)
  [ set pcolor base-pcolor ]
end

;;;;;;;;;;;;;;;;;;;;;;;
;; HubNet Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;

to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ setup-elevator ]
    [
      ifelse hubnet-exit-message?
      [ remove-elevator ]
      [ ask turtles with [ user-id = hubnet-message-source ]
        [ execute-cmd hubnet-message-tag ] ]
    ]
    set received-info? true
  ]
end

to execute-cmd [ cmd ]
  ifelse cmd = "interval-1"
  [ set-velocity 0 ][
  ifelse cmd = "interval-2"
  [ set-velocity 1 ][
  ifelse cmd = "interval-3"
  [ set-velocity 2 ][
  ifelse cmd = "interval-4"
  [ set-velocity 3 ][
  ifelse cmd = "interval-5"
  [ set-velocity 4 ][
  ifelse cmd = "interval-6"
  [ set-velocity 5 ][
  ifelse cmd = "interval-7"
  [ set-velocity 6 ][
  ifelse cmd = "interval-8"
  [ set-velocity 7 ]
  [ set-velocity 8 ] ] ] ] ] ] ] ]
end

to set-velocity [index]
  set velocities replace-item index velocities hubnet-message
end

to setup-elevator
  let p one-of patches with [ pycor = 0 and pxcor > min-pxcor and not any? turtles-on patches with [ pxcor = [pxcor] of myself ] ]
  ifelse p = nobody
  [
    user-message "A user tried to join but there is no more space for another user."
  ]
  [
    crt 1
    [
      set user-id hubnet-message-source
      set velocities [0 0 0 0 0 0 0 0 0]
      set heading 0
      set interval 0
      set color 5 + 10 * random 14
      set ycor-initial random-pycor
      setxy [pxcor] of p ycor-initial
      set label-color yellow
      show-or-hide-id-labels
    ]
  ]
end

to remove-elevator
  ask turtles with [ user-id = hubnet-message-source ] [ die ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plotting Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

;; plot the positions and velocities for the elevators in the appropriate plot
to do-plotting
  ifelse plot-all-elevators?
  [
    let positions [ my-ycor ] of turtles
    let all-velocities [ velocity ] of turtles

    set-current-plot "Elevator Position vs. Intervals"
    plot-list positions false

    set-current-plot "Elevator Velocity vs. Intervals"
    plot-list all-velocities true
    set old-plot-all-elevators? plot-all-elevators?
  ]
  [
    ifelse( elevator-to-plot = nobody )
    [
      if plot-all-elevators? != old-plot-all-elevators?
      [
        user-message (word "If you want to view just 1 elevator's data in the plots, "
                  "please use the PICK ELEVATOR TO PLOT button to pick an "
                  "elevator to plot.\\n"
                  "Alternately, switch PLOT-ALL-ELEVATORS? to true.\\n"
                  "Until you do one of these things, nothing will be plotted.")
        set old-plot-all-elevators? plot-all-elevators?
      ]
    ]
    [
      set-current-plot "Elevator Position vs. Intervals"
      set-current-plot-pen [ user-id ] of elevator-to-plot
      plotxy (clock + 1) [ my-ycor ] of elevator-to-plot

      set-current-plot "Elevator Velocity vs. Intervals"
      set-current-plot-pen [ user-id ] of elevator-to-plot
      plotxy clock [ velocity ] of elevator-to-plot
    ]
  ]
  plot-x-axis "Elevator Velocity vs. Intervals"
  plot-x-axis "Elevator Position vs. Intervals"
end

;; for each element in users, plot the element in info at the same index
to plot-list [ info use-clock? ]
  let index 0

  repeat length users
  [
    set-current-plot-pen item index users  ;; set the pen to be the user-id at index
    ;; plot the data in info at the current index with the current pen
    ifelse use-clock?
    [ plotxy clock item index info ]
    [ plotxy (clock + 1) item index info ]
    set index index + 1
  ]
end

;; plots a black line at the x-axis of the plot this-plot
to plot-x-axis [ this-plot ]
  set-current-plot this-plot
  set-current-plot-pen "x-axis"
  plotxy plot-x-min 0
  plotxy plot-x-max 0
end

;; pick the column of an elevator to show in the plots
;; the plots are not shown unless plot-single? is true
to pick-elevator-to-plot
  if( mouse-down? )
  [
    let x-mouse ( round mouse-xcor )
    let y-mouse ( round mouse-ycor )
    ifelse( count turtles with [ pxcor = x-mouse ] = 1 )
    [
      set elevator-to-plot one-of turtles with [ pxcor = x-mouse ]
      stop
    ]
    [
      ask patch x-mouse y-mouse
      [
        if( count turtles-here = 1 )
        [
          set elevator-to-plot one-of turtles-here
          stop
        ]
      ]
    ]
  ]
end

to-report selected-elevator
  if plot-all-elevators?
   [ report "everybody" ]
  ifelse is-turtle? elevator-to-plot
    [ report [user-id] of elevator-to-plot ]
    [ report "nobody" ]
end

;; setup the position and velocity plot
to setup-plots
  clear-all-plots

  ;; set the list of users to the user-id's of the existing elevators
  set users []

  if any? turtles
  [
    set users [ user-id ] of turtles
    let initial-positions [ ycor-initial ] of turtles

    set-current-plot "Elevator Position vs. Intervals"
    setup-pens false
    plot-list initial-positions true

    set-current-plot "Elevator Velocity vs. Intervals"
    setup-pens true
  ]
end

;; create pens for each of the existing elevators and color the pens to be the same color as
;; their corresponding elevator.  if bars? is true, set the pen mode to be 1 for bar mode.
to setup-pens [ bars? ]
  if any? turtles
  [
    let index 0
    let colors [ color ] of turtles

    repeat length users
    [
      create-temporary-plot-pen item index users
      if bars?
      [ set-plot-pen-mode 1 ]
      set-plot-pen-color item index colors
      set index index + 1
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "Optional- Zoom In (see Tools in the Menu Bar) - optional"
    "Press the LOGIN button."
    "Everyone: Open up a HubNet Client on your machine and..."
    "type your user name, select this activity and press ENTER."
    "Teacher: Ask the students to set the velocity of their elevator at each interval."
    "When all the students are ready press the GO button to run an entire sample..."
    "...or press STEP to run one interval."
    "Teacher: To run the activity again with the same students press the RESET-ELEVATORS button"
    "Optional- Pick a single elevator to watch in the plots and..." "set PLOT-ALL-ELEVATORS? to OFF. - optional"

    "Teacher: To start the simulation over with a new group,..."
    "Have everyone logout of the activity."
    "Press the SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end

;; copyright 2000
@#$#@#$#@
GRAPHICS-WINDOW
187
98
752
414
18
9
15.0
1
12
1
1
1
0
0
0
1
-18
18
-9
9
1
1

CC-WINDOW
5
661
761
756
Command Center
0

BUTTON
95
222
183
255
go
loop [ go if clock = num-intervals [ stop ] ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
42
117
145
150
setup
setup-prompt
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
26
296
164
341
Current Time Interval
clock
0
1
11

BUTTON
5
222
93
255
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
187
11
750
56
Quick Start Instructions (more details in info tab)
quick-start
0
1
11

BUTTON
586
63
664
96
<<< PREV
view-prev
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
666
63
750
96
NEXT >>>
view-next
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
187
62
305
95
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
9
258
180
291
simulation-speed
simulation-speed
0
10
2.6
0.1
1
NIL
HORIZONTAL

PLOT
196
418
470
647
Elevator Position vs. Intervals
Intervals
Position
0.0
9.0
-9.0
9.0
false
false
PENS
"x-axis" 1.0 0 -16777216 true

PLOT
472
418
746
647
Elevator Velocity vs. Intervals
Intervals
Velocity
0.0
9.0
-6.0
6.0
false
false
PENS
"x-axis" 1.0 0 -16777216 true

BUTTON
18
495
187
528
Pick Elevator To Plot
pick-elevator-to-plot
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
18
530
187
563
plot-all-elevators?
plot-all-elevators?
1
1
-1000

BUTTON
29
187
155
220
reset-elevators
re-run
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
10
349
181
382
show-user-id?
show-user-id?
0
1
-1000

BUTTON
42
152
145
185
login
listen-clients
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
43
441
163
486
NIL
selected-elevator
3
1
11

@#$#@#$#@
VERSION
-------
$Id: Elevators.nlogo 37529 2008-01-03 20:38:02Z craig $


WHAT IS IT?
-----------
Each student defines the motion of an elevator by setting its velocity as a function of time on their calculator.  The students have 9 different intervals for which they can set the velocity of the elevator.  They can then send these velocities to NetLogo, where the actions of each elevator can be displayed next to the others.  This can serve as a jumping off point for concepts ranging from derivatives and integrals to wave mechanics.

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------

Quickstart Instructions:
------------------------

Buttons:
--------
SETUP - clears the world including turtles, patches and plots, and draws lines marking the floors when you select this option you should have all the students logout of the activity and log back in.
RESET-ELEVATORS - sets the clock to zero, returns all the elevators to their initial positions and clears the plots.
STEP - moves the elevators by 1 interval
GO -  moves the elevators 9 intervals.

PICK ELEVATOR TO PLOT - allows the user to pick an elevator to plot the velocity and position information for.

NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

Sliders:
--------
SIMULATION-SPEED - The higher the number the faster the activity runs.

Switches:
---------
PLOT-ALL-ELEVATORS? - if true, the data for all the elevators will be shown in the plots.  If it is false and if an elevator has been picked by using the PICK ELEVATOR TO PLOT button, only the picked elevator's information will be shown in the plot from the current time tick forward.  Old plot data is not erased.
SHOW-USER-ID? - if true, all the turtles show their user-id as its label.  If false, no label is shown.


THINGS TO NOTICE
----------------
Notice if an elevator moves more than 1 floor during an interval, it equally divides the amount of time spent traveling through each floor.


THINGS TO TRY
-------------
Identify your elevator (necessary even with the first use!)
Have your elevator sit at the third floor during the fourth time segment.  What is the same / different between the solutions?
Make a traveling / standing wave.
Have a velocity of +2 during the fourth time interval.
Start and End at the same floor.


EXTENDING THE MODEL
-------------------
A real elevator does not move at a constant speed when moving from floor to floor.  It will build its speed gradually until it has reached a constant speed.  Then before reaching the floor it is supposed to stop on, it will slow down.  Try to make this happen as a part of the model.


CREDITS AND REFERENCES
----------------------
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

elevator
false
0
Rectangle -1 true false 45 14 255 286
Circle -7500403 true true 115 16 69
Rectangle -7500403 true true 139 82 159 101
Rectangle -7500403 true true 109 92 192 206
Polygon -7500403 true true 191 109 248 127 234 164 190 128
Polygon -7500403 true true 109 109 53 131 62 162 109 126
Rectangle -7500403 true true 118 205 146 278
Rectangle -7500403 true true 157 205 181 277
Rectangle -7500403 true true 180 264 203 277
Rectangle -7500403 true true 92 265 118 278

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.0pre8
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
16
14
571
299
18
9
15.0
1
12
1
1
1
0
0
0
1
-18
18
-9
9

SLIDER
117
312
154
462
interval-1
interval-1
-5
5
0
1
1
NIL
VERTICAL

SLIDER
155
312
192
462
interval-2
interval-2
-5
5
0
1
1
NIL
VERTICAL

SLIDER
193
312
230
462
interval-3
interval-3
-5
5
0
1
1
NIL
VERTICAL

SLIDER
231
312
268
462
interval-4
interval-4
-5
5
0
1
1
NIL
VERTICAL

SLIDER
269
312
306
462
interval-5
interval-5
-5
5
0
1
1
NIL
VERTICAL

SLIDER
307
312
344
462
interval-6
interval-6
-5
5
0
1
1
NIL
VERTICAL

SLIDER
345
312
382
462
interval-7
interval-7
-5
5
0
1
1
NIL
VERTICAL

SLIDER
383
312
420
462
interval-8
interval-8
-5
5
0
1
1
NIL
VERTICAL

SLIDER
421
312
458
462
interval-9
interval-9
-5
5
0
1
1
NIL
VERTICAL

@#$#@#$#@
