
;; These are global variables:
globals [
  ;; The following two boolean variables take the values 0 and 1 instead of true and false because they must be able
  ;; to generate error messages even if they have not been initialized.
  positions-made?  ;; positions must be made before vacancies can be made;
  vacancies-made?  ;; vacancies must be made before vacancy chains can be made;
  wall-length  ;; is the length of walls;
  meter  ;; measures the length of walls;
  direction  ;; tells where walls are expanded;
  x-target  ;; the x-coordinate of a position to be reached;
  y-target  ;; the y-coordinate of a position to be reached;
  reachable?  ;; whether the target position can be actually reached;
  x y  ;; scan the screen in search for patches where new turtles can be posed;
  cum-num-chains-by-length-list  ;; a list for the cumulative number of chains, where each element refers to a different chain length
  max-chain-length-attained  ;; the maximum length vacancy chains actually reached.
]

;; These variables belong to patches:
patches-own [elevation]  ;; the stratum of a patch, with 0 as the lowest stratum.


;; These variables belong to turtles:
turtles-own [cl]  ;; the length of a vacancy chain.







;; Repeats the sequence "setup-hierarchy" - "setup-vacancies" - "go" so many times as the parameter "times".
;; At the end, it prints on the Command Center the average number of chains for each chain length.
to iterate
  ;; The list is filled up with values. Counters are in order because this list may entail several instances of the same value.
  repeat times [
    setup-positions
    setup-vacancies
    go
    ;; The list of the cumulative number of chains of each length is initialized.
    set cum-num-chains-by-length-list []
    repeat max-chain-length
      [set cum-num-chains-by-length-list fput 0 cum-num-chains-by-length-list]
    ;; Introduces a counter.
    let i 0
    ;; Updates the list of the cumulative number of chains of each length.
    foreach cum-num-chains-by-length-list [
      set cum-num-chains-by-length-list replace-item i cum-num-chains-by-length-list (? + count turtles with [cl = i + 1])
      set i (i + 1)
    ]
  ]
  ;; The list is stripped of all last 0 values.
  set cum-num-chains-by-length-list reverse cum-num-chains-by-length-list
  let first-non-zero-value-found? false
  foreach cum-num-chains-by-length-list [
    ifelse (? = 0 and first-non-zero-value-found? = false)
      [set cum-num-chains-by-length-list remove ? cum-num-chains-by-length-list]
      [set first-non-zero-value-found? true]
  ]
  set cum-num-chains-by-length-list reverse cum-num-chains-by-length-list
  set max-chain-length-attained length cum-num-chains-by-length-list
  ;; Output is printed on the screen
  let i 0
  foreach cum-num-chains-by-length-list [
    set i (i + 1)
    type "Percent of chains of length "
    type i
    type ": "
    ifelse (sum cum-num-chains-by-length-list > 0)
      [print ? * 100 / sum cum-num-chains-by-length-list]
      [print 0]
  ]
  print ""
  let cum-num-chains-by-length-except-length-one-list remove-item 0 cum-num-chains-by-length-list
  type "Number of chains of length greater than 1: "
  print (sum cum-num-chains-by-length-except-length-one-list) / times
  type "Percent of chains of length greater than 1: "
  ifelse (sum cum-num-chains-by-length-list > 0)
    [print (sum cum-num-chains-by-length-except-length-one-list) * 100 / (sum cum-num-chains-by-length-list)]
    [print 0]
  print ""
  print ""
  print ""
end



to setup-positions
  clear-all
  ;; Patches are assigned an "elevation" either according to a decreasing exponential or a uniform or an increasing exponential
  ;; distribution. Elevation takes integer values.
  ;; The elevation is reflected by a gray scale. Patches with highest elevation are white, patches with lowest elevation are black.
  ask turtles [die]
  ask patches [
    ;; Decreasing exponential distribution.
    if (strata-distribution = "hierarchical") [
      set elevation n-of-strata + 1
      while [elevation > n-of-strata]
        [set elevation round random-exponential ((1 + n-of-strata) / 4)]
    ]
    ;; The complement of a decreasing exponential distribution.
    if (strata-distribution = "inverse") [
      set elevation -1
      while [elevation < 0]
        [set elevation n-of-strata - round random-exponential ((1 + n-of-strata) / 4)]
    ]
    ;; Uniform distribution.
    if (strata-distribution = "linear")
      [set elevation random (1 + n-of-strata)]
    ;; Mixed: decreasing exponential for high strata, the complement of a poisson distribution for low strata.
    if (strata-distribution = "lean") [
      ifelse (random 2 = 0) [
        set elevation n-of-strata + 1
        while [elevation > n-of-strata]
          [set elevation round random-exponential (1 + n-of-strata) / 4]
      ][
        set elevation -1
        while [elevation < 0]
          [set elevation n-of-strata - round random-poisson (1 + n-of-strata) / 4]
      ]
    ]
    if (elevation <= n-of-strata)
      [set pcolor scale-color black elevation 0 n-of-strata]
  ]
  ;; Builds walls.
  if (n-walls > 0) [
    while [count patches with [pcolor = red] < n-walls] [
      ask patch (random-pxcor) (random-pycor) [
        set pcolor red
        set elevation n-of-strata + 1
      ]
    ]
    ask patches with [pcolor = red] [
      set wall-length (random max (list max-pxcor max-pycor) + 1)
      if (wall-length > 1) [
        set meter 1
        set direction random 360
        add-brick
      ]
    ]
  ]
  ;; Global variables keep track of the fact that positions have been made, whereas vacancies are still waiting.
  set positions-made? 1
  set vacancies-made? 0
end
;; Bricks are added to walls. This procedure is called by "setup-positions".
to add-brick
  let next-brick patch-at-heading-and-distance direction 1
  let reach 1
  while [[pcolor] of next-brick = red and any? patches with [pcolor != red]] [
    set reach reach + 1
    if (random 10 <= 1)
      [set direction random 360]
    set next-brick patch-at-heading-and-distance direction reach
  ]
  ask next-brick [
    set pcolor red
    set elevation n-of-strata + 1
    set meter meter + 1
    if (meter < wall-length)
      [add-brick]
  ]
end



to setup-vacancies
  ifelse (starting-stratum > n-of-strata) [
    print "The starting stratum cannot be greater than the number of strata."
  ][
    ifelse (positions-made? = 0) [
      print "Please press the Setup Positions button."
    ][       
      ;; Two nested WHILEs in order to scan the screen.
      ;; If a free patch of the required elevation is found, a new turtle is created and posed there.
      set x min-pxcor
      while [x <= max-pxcor] [
        set y min-pycor
        while [y <= max-pycor] [
          if ([elevation] of patch x y = starting-stratum) [
            set-default-shape turtles "vacancy"
            crt 1 [
              set color green
              set xcor x
              set ycor y
              set cl 0
            ]
          ]
          set y (y + 1)
        ]
        set x (x + 1)
      ]
      ;; A global variable keeps track of the fact that vacancies have been made available.
      set vacancies-made? 1
    ]
  ]
  reset-ticks
end



to go
  ifelse (starting-stratum > n-of-strata) [
    print "The starting stratum cannot be greater than the number of strata."
  ][
    if (positions-made? = 0) [print "Please press the 'Setup Positions' button."]
    if (vacancies-made? = 0) [print "Please press the 'Setup Vacancies' button."]
    if (positions-made? = 1 and vacancies-made? = 1) [
      ask turtles [form-chains]
      ;; Vacancy chains cannot be formed again.
      set positions-made? 0
      set vacancies-made? 0
    ]
  ]
  tick
end



to form-chains
  ;; This procedure calls itself recursively with decreasing values of elevation until two or more patches with equal elevation are found.
  ;; Elevation in neighboring patches. Angles are measured clockwise from pi/2.
  ;; Turtles are the heads of vacancy chains. Eventually, turtles move.
  ;; First, they look for a patch among the closest neighbors (4- or 8- or 24-neighborhood). In a second turn, they will look among the second-tier neighbors (24-neighborhood).
  ;; Each time the procedure "form-chains" is called, the chain lenght is increased by one unit.
  set cl (cl + 1)
  ;; If the 4-neighborhood has been chosen. 
  if (neighborhood = 4) [
    ;; Look north.
    ifelse (any? turtles-on patch-at 0 1 = false and ([elevation] of patch-at 0 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1)) [
      ask patch-here [set pcolor yellow]
      set ycor ycor + 1
      form-chains
    ][
      ;; Look east. 
      ifelse (any? turtles-on patch-at 1 0 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1)) [
        ask patch-here [set pcolor yellow]
        set xcor xcor + 1
        form-chains
      ][
        ;; Look south.
        ifelse (any? turtles-on patch-at 0 -1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1)) [
          ask patch-here [set pcolor yellow]
          set ycor ycor - 1
          form-chains
        ][
          ;; Look west.   
          ifelse (any? turtles-on patch-at -1 0 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 = [elevation] of patch-here - 1)) [
            ask patch-here [set pcolor yellow]
            set xcor xcor - 1
            form-chains
          ][
            if (verbose?) [
              type "Vacancy "
              type who
              type " produced a chain of length "
              print cl
            ]
          ]
        ]
      ]
    ]
  ]
  ;; If the 8-neighborhood has been chosen.
  if (neighborhood = 8) [
    ;; Look north.
    ifelse (any? turtles-on patch-at 0 1 = false and ([elevation] of patch-at 0 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
      ask patch-here [set pcolor yellow]
      set ycor ycor + 1
      form-chains
    ][
      ;; Look north-east.
      ifelse (any? turtles-on patch-at 1 1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
        ask patch-here [set pcolor yellow]
        setxy (xcor + 1) (ycor + 1)
        form-chains
      ][
        ;; Look east. 
        ifelse (any? turtles-on patch-at 1 0 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
          ask patch-here [set pcolor yellow]
          set xcor xcor + 1
          form-chains
        ][
          ;; Look south-east.
          ifelse (any? turtles-on patch-at 1 -1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
            ask patch-here [set pcolor yellow]
            setxy (xcor + 1) (ycor - 1)
            form-chains
          ][
            ;; Look south.
            ifelse (any? turtles-on patch-at 0 -1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
              ask patch-here [set pcolor yellow]
              set ycor ycor - 1
              form-chains
            ][
              ;; Look south-west.
              ifelse (any? turtles-on patch-at -1 -1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
                ask patch-here [set pcolor yellow]
                setxy (xcor - 1) (ycor - 1)
                form-chains
              ][
                ;; Look west.   
                ifelse (any? turtles-on patch-at -1 0 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1)) [
                  ask patch-here [set pcolor yellow]
                  set xcor xcor - 1
                  form-chains
                ][  
                  ;; Look north-west.
                  ifelse (any? turtles-on patch-at -1 1 = false and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 = [elevation] of patch-here - 1)) [
                    ask patch-here [set pcolor yellow]
                    setxy (xcor - 1) (ycor + 1)
                    form-chains
                  ][
                    if (verbose?) [
                      type "Vacancy "
                      type who
                      type " produced a chain of length "
                      print cl
                    ]
                  ]
                ]
              ]
            ]
          ]
        ]
      ]
    ]
  ]
  ;; If the 24-neighborhood has been chosen.
  if (neighborhood = 24) [
    ;; Look north.
    ;; North, first tier.
    set x-target xcor
    set y-target ycor + 1
    check-reachability
    ifelse (any? (turtles-on patch-at 0 1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
      ask patch-here [set pcolor yellow]
      set xcor xcor + 1
      form-chains
    ][
      ;; North, second tier.
      set x-target xcor
      set y-target ycor + 2
      check-reachability
      ifelse (any? (turtles-on patch-at 0 2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
        ask patch-here [set pcolor yellow]
        set xcor xcor + 2
        form-chains
      ][
        ;; Look north-northeast (only the second tier).
        set x-target xcor + 1
        set y-target ycor + 2
        check-reachability
        ifelse (any? (turtles-on patch-at 1 2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 = [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
          ask patch-here [set pcolor yellow]
          setxy (xcor + 1) (ycor + 2)
          form-chains
        ][
          ;; Look north-east.
          ;; North-East, first tier.
          set x-target xcor + 1
          set y-target ycor + 1
          check-reachability
          ifelse (any? (turtles-on patch-at 1 1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
            ask patch-here [set pcolor yellow]
            setxy (xcor + 1) (ycor + 1)
            form-chains
          ][
            ;; North-East, second tier.
            set x-target xcor + 2
            set y-target ycor + 2
            check-reachability
            ifelse (any? (turtles-on patch-at 2 2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 = [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
              ask patch-here [set pcolor yellow]
              setxy (xcor + 2) (ycor + 2)
              form-chains
            ][
              ;; Look east-northeast (only the second tier).
              set x-target xcor + 2
              set y-target ycor + 1
              check-reachability
              ifelse (any? (turtles-on patch-at 2 1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                ask patch-here [set pcolor yellow]
                setxy (xcor + 2) (ycor + 1)
                form-chains
              ][
                ;; Look east.
                ;; East, first tier.
                set x-target xcor + 1
                set y-target ycor
                check-reachability
                ifelse (any? (turtles-on patch-at 1 0) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                  ask patch-here [set pcolor yellow]
                  set xcor xcor + 1
                  form-chains
                ][
                  ;; East, second tier.
                  set x-target xcor + 2
                  set y-target ycor
                  check-reachability
                  ifelse (any? (turtles-on patch-at 2 0) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                    ask patch-here [set pcolor yellow]
                    set xcor xcor + 2
                    form-chains
                  ][
                    ;; Look east-southeast (only the second tier).
                    set x-target xcor + 2
                    set y-target ycor - 1
                    check-reachability
                    ifelse (any? (turtles-on patch-at 2 -1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                      ask patch-here [set pcolor yellow]
                      setxy (xcor + 2) (ycor - 1)
                      form-chains
                    ][
                      ;; Look southeast.
                      ;; South-East, first tier.
                      set x-target xcor + 1
                      set y-target ycor - 1
                      check-reachability
                      ifelse (any? (turtles-on patch-at 1 -1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                        ask patch-here [set pcolor yellow]
                        setxy (xcor + 1) (ycor - 1)
                        form-chains
                      ][
                        ;; South-East, second tier.
                        set x-target xcor + 2
                        set y-target ycor - 2
                        check-reachability
                        ifelse (any? (turtles-on patch-at 2 -2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 = [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                          ask patch-here [set pcolor yellow]
                          setxy (xcor + 2) (ycor - 2)
                          form-chains
                        ][
                          ;; Look south-southeast (only the second tier).
                          set x-target xcor + 1
                          set y-target ycor - 2
                          check-reachability
                          ifelse (any? (turtles-on patch-at 1 -2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 = [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                            ask patch-here [set pcolor yellow]
                            setxy (xcor + 1) (ycor - 2)
                            form-chains
                          ][
                            ;; Look south.
                            ;; South, first tier.
                            set x-target xcor
                            set y-target ycor - 1
                            check-reachability
                            ifelse (any? (turtles-on patch-at 0 -1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                              ask patch-here [set pcolor yellow]
                              set ycor ycor - 1
                              form-chains
                            ][
                              ;; South, second tier.
                              set x-target xcor
                              set y-target ycor - 2
                              check-reachability
                              ifelse (any? (turtles-on patch-at 0 -2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                ask patch-here [set pcolor yellow]
                                set ycor ycor - 2
                                form-chains
                              ][
                                ;; Look south-southwest (only the second tier).
                                set x-target xcor - 1
                                set y-target ycor - 2
                                check-reachability
                                ifelse (any? (turtles-on patch-at -1 -2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 = [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                  ask patch-here [set pcolor yellow]
                                  setxy (xcor - 1) (ycor - 2)
                                  form-chains
                                ][
                                  ;; Look south-west.
                                  ;; South-West, first tier.
                                  set x-target xcor - 1
                                  set y-target ycor - 1
                                  check-reachability
                                  ifelse (any? (turtles-on patch-at -1 -1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                    ask patch-here [set pcolor yellow]
                                    setxy (xcor - 1) (ycor - 1)
                                    form-chains
                                  ][
                                    ;; South-West, second tier.
                                    set x-target xcor - 2
                                    set y-target ycor - 2
                                    check-reachability
                                    ifelse (any? (turtles-on patch-at -2 -2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 = [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                      ask patch-here [set pcolor yellow]
                                      setxy (xcor - 2) (ycor - 2)
                                      form-chains
                                    ][
                                      ;; Look west-southwest (only the second tier).
                                      set x-target xcor - 2
                                      set y-target ycor - 1
                                      check-reachability
                                      ifelse (any? (turtles-on patch-at -2 -1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                        ask patch-here [set pcolor yellow]
                                        setxy (xcor - 2) (ycor - 1)
                                        form-chains
                                      ][
                                        ;; Look west.
                                        ;; West, first tier.
                                        set x-target xcor - 1
                                        set y-target ycor
                                        check-reachability
                                        ifelse (any? (turtles-on patch-at -1 0) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                          ask patch-here [set pcolor yellow]
                                          set xcor xcor - 1
                                          form-chains
                                        ][
                                          ;; West, second tier.
                                          set x-target xcor - 2
                                          set y-target ycor
                                          check-reachability
                                          ifelse (any? (turtles-on patch-at -2 0) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 = [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                            ask patch-here [set pcolor yellow]
                                            set xcor xcor - 2
                                            form-chains
                                          ][
                                            ;; Look west-northwest (only the second tier).
                                            set x-target xcor - 2
                                            set y-target ycor + 1
                                            check-reachability
                                            ifelse (any? (turtles-on patch-at -2 1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                              ask patch-here [set pcolor yellow]
                                              setxy (xcor - 2) (ycor + 1)
                                              form-chains
                                            ][
                                              ;; Look northwest.
                                              ;; North-West, first tier.
                                              set x-target xcor - 1
                                              set y-target ycor + 1
                                              check-reachability
                                              ifelse (any? (turtles-on patch-at -1 1) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 = [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                                ask patch-here [set pcolor yellow]
                                                setxy (xcor - 1) (ycor + 1)
                                                form-chains
                                              ][
                                                ;; North-West, second tier.
                                                set x-target xcor - 2
                                                set y-target ycor + 2
                                                check-reachability
                                                ifelse (any? (turtles-on patch-at -2 2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 = [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 != [elevation] of patch-here - 1)) [
                                                  ask patch-here [set pcolor yellow]
                                                  setxy (xcor - 2) (ycor + 2)
                                                  form-chains
                                                ][
                                                  ;; Look north-northwest (only the second tier).
                                                  set x-target xcor - 1
                                                  set y-target ycor + 2
                                                  check-reachability
                                                  ifelse (any? (turtles-on patch-at -1 2) with [color = green] = false and reachable? and ([elevation] of patch-at 0 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at 2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at 0 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 -1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 0 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 1 != [elevation] of patch-here - 1) and ([elevation] of patch-at -2 2 != [elevation] of patch-here - 1) and ([elevation] of patch-at -1 2 = [elevation] of patch-here - 1)) [
                                                    ask patch-here [set pcolor yellow]
                                                    setxy (xcor - 1) (ycor + 2)
                                                    form-chains
                                                  ][
                                                    if (verbose?) [
                                                      type "Vacancy "
                                                      type who
                                                      type " produced a chain of length "
                                                      print cl
                                                    ]
                                                  ]
                                                ]
                                              ]
                                            ]
                                          ]
                                        ]
                                      ]
                                    ]
                                  ]
                                ]
                              ]
                            ]
                          ]
                        ]
                      ]
                    ]
                  ]
                ]
              ]
            ]
          ]
        ]
      ]
    ]
  ]
  ;; Although the turtles at the head of vacancy chains are green, showing a green patch looks nicer.
  ask patch-here
    [set pcolor green]
end
;; This procedure checks that a position in the 24-neighborhood is not forbidden by a wall.
to check-reachability
  set reachable? false
  foreach sort neighbors [
    if ([pcolor] of ? != red and any? turtles-on ? = false) [
      hatch 1 [
        setxy [pxcor] of ? [pycor] of ?
        set color blue
        set hidden? true
        repeat 20 [
          set heading random 360
          if ([pcolor] of patch-ahead 1 != red and member? (patch-ahead 1) ([patches in-radius 3] of myself)) [
            setxy ([pxcor] of patch-ahead 1) ([pycor] of patch-ahead 1)
            if (xcor = x-target and ycor = y-target) [
              set reachable? true
              stop
            ]
          ]
        ]
        die
      ]
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
356
10
726
308
15
11
11.613
1
10
1
1
1
0
1
1
1
-15
15
-11
11
0
0
1
ticks
30.0

BUTTON
10
10
141
44
Setup Positions
setup-positions
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
275
10
349
44
Go
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
9
174
349
308
Length  Distribution
Length
No. of chains
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 false "" "histogram [cl] of turtles"

SLIDER
120
89
233
122
n-of-strata
n-of-strata
1
10
5
1
1
NIL
HORIZONTAL

CHOOSER
10
123
122
168
starting-stratum
starting-stratum
0 1 2 3 4 5 6 7 8 9 10
4

CHOOSER
234
123
350
168
strata-distribution
strata-distribution
"hierarchical" "inverse" "linear" "lean"
0

CHOOSER
121
123
234
169
neighborhood
neighborhood
4 8 24
0

BUTTON
140
10
276
44
Setup Vacancies
Setup-vacancies
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
9
89
121
122
verbose?
verbose?
1
1
-1000

BUTTON
9
50
65
84
Iterate
iterate
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
65
50
158
84
times
times
10
1000
10
10
1
NIL
HORIZONTAL

SLIDER
233
89
349
122
n-walls
n-walls
0
100
0
1
1
NIL
HORIZONTAL

SLIDER
158
50
276
84
max-chain-length
max-chain-length
0
100
10
1
1
NIL
HORIZONTAL

MONITOR
275
44
349
89
max-attained
max-chain-length-attained
17
1
11

@#$#@#$#@

## WHAT IS IT?

Vacancy chains are a means of resource allocation alternative to markets or other forms of competition. Vacancy chains take place if the resource is sufficiently specific and the information sufficiently restricted for there being only one applicant to the resource. In fact, if there are two or more applicants a competition sets in, and eventually a market may arise in order to manage this competition.

Quite often, organizations restrict information and require specificity in order to allocate resources by means of vacancy chains. For instance, positions can be made very specific and information on available positions can be made hardly available so careers happen because the retirement of a high-ranked official triggers a chain of promotions among his subordinates. Another instance may be the way the most expensive and unique houses are allocated. Similarly, vacancy chains have been observed among hermit crabs lining up for occupying empty shells. Decentralized allocation of problems among robots can also be achieved by means of vacancy chains.

The vacancy chains model considers a set of positions ordered in strata. If a vacancy is produced, the corresponding position is occupied by agents from lower strata. Thus, vacancy chains propagate from upper to lower strata.


## HOW IT WORKS

The model landscape represents positions in an organization. Each position is characterized by an "elevation" representing the stratum to which it belongs - strata may represent levels in a hierarchy, size of houses or shells, etc. The positions of stratum zero represent the surrounding competitive arena. The highest stratum is white, the lowest stratum is black, intermediate strata are depicted according to different shades of grey.

Vacancies appear as green areas. If they find just one neighbor exactly one stratum lower than them, they diffuse down the ladder leaving a yellow stripe behind them. Thus, vacancy chains appear as yellow worms with a green head.

As soon as a vacancy finds two or more neighbors exactly one stratum lower than itself, the vacancy chain stops. This is the point where competition begins. It may happen within an organization, or at the border between an organization and the surrounding market.

Vacancy chains may be influenced by the size of organizations, as well as by extent of their connections. This model allows to build "walls" in organization space that halt the propagation of vacancy chains. Walls generate islands of larger portions of positions eventually linked by tiny bridges, mimicking the formation of network organizations.


## HOW TO USE IT

The button "Setup Positions" assigns positions to strata. The slider "n-walls" selects the number of walls that separate positions from one another.

The number of strata is set by means of the slider "n-of-strata". The distribution of strata is chosen by means of the selector "strata-distribution", which can be:  
1) "hierarchical": A hierarchy characterized by an exponential distribution of position, i.e. exponentially more positions have been assigned to lower strata;  
2) "inverse": An inverted hierarchy characterized by an exponential distribution of positions, i.e. exponentially more positions have been assigned to upper strata;  
3) "linear": A linear organization characterized by a uniform distribution, i.e. there are just as many positions in the lower as in the upper strata;  
4) "lean": A hierarchy characterized by an exponential distribution of the upper positions and a poisson distribution of the lower strata, i.e. such that middle positions are far less than in the hierarchy produced by an exponential distribution.

The slider "starting stratum" allows to select the stratum where vacancies will be created. The button "Setup Vacancies" creates vacancies at the desired stratum. An error message is issued if the starting stratum is larger than the number of strata.

The button "Go" creates vacancy chains. Chains form depending on the size of the neighborhood that observes a vacancy. The model allows to select a neighborhood of 4, 8 or 24 positions around a vacancy.

If the "verbose?" option is selected, the model writes on the Command Center which vacancies are forming chains, and how long they are. Note that an isolated vacancy is a chain of length 1.

The button "iterate" repeats the sequence "Setup Positions" - "Setup Vacancies" - "Go" a number of times specified by "times". The slider "max-chain-length" specifies the maximum length of chains that will be analyzed. The monitor "max-attained" tells which is the maximum chain length that has been actually attained. In order for "iterate" to work properly, "max-chain-length" must be greater than "max-attained".
At the end, the following information is written on the command center:  
- The percent of chains of each length;  
- The number and percent of chains of length greater than one.  
All information is averaged over the number of runs specified by "times". 


## THINGS TO NOTICE

The larger the neighborhood, the more likely that an agent able to fill a vacancy will found. However, the larger the neighborhood, the more likely that two or more agents able to fill a vacancy are found. Thus, large neighborhoods lengthen or shorten chains depending on starting stratum and organizational form.

The functioning of the model is clearest if the number of strata is small, i.e. 2 or 3. Then, it is easy to see that even if positions of the required level are available in the neighborhood of a vacancy, the chain stops if the neighborhood entails more than one such vacancy.


## THINGS TO TRY

Depending on organizational form, certain configurations of parameters are such that the proportion of chains of length greater than one increases with the size of the neighborhood, others are such that the proportion of chains of length greater than one decreases with the size of the neighborhood. A few configurations are such that the proportion of chains of length greater than one increases when the neighborhood increases from 4 to 8, decreases when the neighborhood increases from 8 to 24. The first two regions correspond to connected areas in parameters space, whereas the third region is a thin border between them.


## EXTENDING THE MODEL

This model focuses on vacancy chains as a means for resource allocation. However, even after a vacancy chains loses this function to a market or some other form of competition, it can still be tracked. A model that would also allow for resource allocation by competition would enable the user to observe vacancy chains even after they lost their function as a means for the allocation of scarce resources.


## NETLOGO FEATURES

The procedure "form-chains" calls itself recursively. It forms chains by moving vacancies down positions of decreasing elevation.


## CREDITS AND REFERENCES

This model is thoroughly described in:

Guido Fioretti  
A Model of Vacancy Chains as a Mechanism for Resource Allocation  
Journal of Mathematical Sociology, 34 (1) 2010: 52-75.

This model was built by:

Guido Fioretti  
University of Bologna  
Department of Management Science  
www.cs.unibo.it/fioretti
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

ant
true
0
Polygon -7500403 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7500403 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7500403 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7500403 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7500403 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7500403 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7500403 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7500403 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7500403 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee
true
0
Polygon -1184463 true false 152 149 77 163 67 195 67 211 74 234 85 252 100 264 116 276 134 286 151 300 167 285 182 278 206 260 220 242 226 218 226 195 222 166
Polygon -16777216 true false 150 149 128 151 114 151 98 145 80 122 80 103 81 83 95 67 117 58 141 54 151 53 177 55 195 66 207 82 211 94 211 116 204 139 189 149 171 152
Polygon -7500403 true true 151 54 119 59 96 60 81 50 78 39 87 25 103 18 115 23 121 13 150 1 180 14 189 23 197 17 210 19 222 30 222 44 212 57 192 58
Polygon -16777216 true false 70 185 74 171 223 172 224 186
Polygon -16777216 true false 67 211 71 226 224 226 225 211 67 211
Polygon -16777216 true false 91 257 106 269 195 269 211 255
Line -1 false 144 100 70 87
Line -1 false 70 87 45 87
Line -1 false 45 86 26 97
Line -1 false 26 96 22 115
Line -1 false 22 115 25 130
Line -1 false 26 131 37 141
Line -1 false 37 141 55 144
Line -1 false 55 143 143 101
Line -1 false 141 100 227 138
Line -1 false 227 138 241 137
Line -1 false 241 137 249 129
Line -1 false 249 129 254 110
Line -1 false 253 108 248 97
Line -1 false 249 95 235 82
Line -1 false 235 82 144 100

bird1
false
0
Polygon -7500403 true true 2 6 2 39 270 298 297 298 299 271 187 160 279 75 276 22 100 67 31 0

bird2
false
0
Polygon -7500403 true true 2 4 33 4 298 270 298 298 272 298 155 184 117 289 61 295 61 105 0 43

boat1
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 33 230 157 182 150 169 151 157 156
Polygon -7500403 true true 149 55 88 143 103 139 111 136 117 139 126 145 130 147 139 147 146 146 149 55

boat2
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 157 54 175 79 174 96 185 102 178 112 194 124 196 131 190 139 192 146 211 151 216 154 157 154
Polygon -7500403 true true 150 74 146 91 139 99 143 114 141 123 137 126 131 129 132 139 142 136 126 142 119 147 148 147

boat3
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7500403 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

butterfly1
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7500403 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7500403 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7500403 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7500403 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7500403 true 167 47 159 82
Line -7500403 true 136 47 145 81
Circle -7500403 true true 165 45 8
Circle -7500403 true true 134 45 6
Circle -7500403 true true 133 44 7
Circle -7500403 true true 133 43 8

circle
false
0
Circle -7500403 true true 35 35 230

person
false
0
Circle -7500403 true true 155 20 63
Rectangle -7500403 true true 158 79 217 164
Polygon -7500403 true true 158 81 110 129 131 143 158 109 165 110
Polygon -7500403 true true 216 83 267 123 248 143 215 107
Polygon -7500403 true true 167 163 145 234 183 234 183 163
Polygon -7500403 true true 195 163 195 233 227 233 206 159

sheep
false
15
Rectangle -1 true true 90 75 270 225
Circle -1 true true 15 75 150
Rectangle -16777216 true false 81 225 134 286
Rectangle -16777216 true false 180 225 238 285
Circle -16777216 true false 1 88 92

spacecraft
true
0
Polygon -7500403 true true 150 0 180 135 255 255 225 240 150 180 75 240 45 255 120 135

thin-arrow
true
0
Polygon -7500403 true true 150 0 0 150 120 150 120 293 180 293 180 150 300 150

truck-down
false
0
Polygon -7500403 true true 225 30 225 270 120 270 105 210 60 180 45 30 105 60 105 30
Polygon -8630108 true false 195 75 195 120 240 120 240 75
Polygon -8630108 true false 195 225 195 180 240 180 240 225

truck-left
false
0
Polygon -7500403 true true 120 135 225 135 225 210 75 210 75 165 105 165
Polygon -8630108 true false 90 210 105 225 120 210
Polygon -8630108 true false 180 210 195 225 210 210

truck-right
false
0
Polygon -7500403 true true 180 135 75 135 75 210 225 210 225 165 195 165
Polygon -8630108 true false 210 210 195 225 180 210
Polygon -8630108 true false 120 210 105 225 90 210

turtle
true
0
Polygon -7500403 true true 138 75 162 75 165 105 225 105 225 142 195 135 195 187 225 195 225 225 195 217 195 202 105 202 105 217 75 225 75 195 105 187 105 135 75 142 75 105 135 105

vacancy
false
6
Polygon -13840069 true true 45 255 255 255 255 45 45 45

wolf
false
0
Rectangle -7500403 true true 15 105 105 165
Rectangle -7500403 true true 45 90 105 105
Polygon -7500403 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7500403 true true 105 120 263 195
Rectangle -7500403 true true 108 195 259 201
Rectangle -7500403 true true 114 201 252 210
Rectangle -7500403 true true 120 210 243 214
Rectangle -7500403 true true 115 114 255 120
Rectangle -7500403 true true 128 108 248 114
Rectangle -7500403 true true 150 105 225 108
Rectangle -7500403 true true 132 214 155 270
Rectangle -7500403 true true 110 260 132 270
Rectangle -7500403 true true 210 214 232 270
Rectangle -7500403 true true 189 260 210 270
Line -7500403 true 263 127 281 155
Line -7500403 true 281 155 281 192

wolf-left
false
3
Polygon -6459832 true true 117 97 91 74 66 74 60 85 36 85 38 92 44 97 62 97 81 117 84 134 92 147 109 152 136 144 174 144 174 103 143 103 134 97
Polygon -6459832 true true 87 80 79 55 76 79
Polygon -6459832 true true 81 75 70 58 73 82
Polygon -6459832 true true 99 131 76 152 76 163 96 182 104 182 109 173 102 167 99 173 87 159 104 140
Polygon -6459832 true true 107 138 107 186 98 190 99 196 112 196 115 190
Polygon -6459832 true true 116 140 114 189 105 137
Rectangle -6459832 true true 109 150 114 192
Rectangle -6459832 true true 111 143 116 191
Polygon -6459832 true true 168 106 184 98 205 98 218 115 218 137 186 164 196 176 195 194 178 195 178 183 188 183 169 164 173 144
Polygon -6459832 true true 207 140 200 163 206 175 207 192 193 189 192 177 198 176 185 150
Polygon -6459832 true true 214 134 203 168 192 148
Polygon -6459832 true true 204 151 203 176 193 148
Polygon -6459832 true true 207 103 221 98 236 101 243 115 243 128 256 142 239 143 233 133 225 115 214 114

wolf-right
false
3
Polygon -6459832 true true 170 127 200 93 231 93 237 103 262 103 261 113 253 119 231 119 215 143 213 160 208 173 189 187 169 190 154 190 126 180 106 171 72 171 73 126 122 126 144 123 159 123
Polygon -6459832 true true 201 99 214 69 215 99
Polygon -6459832 true true 207 98 223 71 220 101
Polygon -6459832 true true 184 172 189 234 203 238 203 246 187 247 180 239 171 180
Polygon -6459832 true true 197 174 204 220 218 224 219 234 201 232 195 225 179 179
Polygon -6459832 true true 78 167 95 187 95 208 79 220 92 234 98 235 100 249 81 246 76 241 61 212 65 195 52 170 45 150 44 128 55 121 69 121 81 135
Polygon -6459832 true true 48 143 58 141
Polygon -6459832 true true 46 136 68 137
Polygon -6459832 true true 45 129 35 142 37 159 53 192 47 210 62 238 80 237
Line -16777216 false 74 237 59 213
Line -16777216 false 59 213 59 212
Line -16777216 false 58 211 67 192
Polygon -6459832 true true 38 138 66 149
Polygon -6459832 true true 46 128 33 120 21 118 11 123 3 138 5 160 13 178 9 192 0 199 20 196 25 179 24 161 25 148 45 140
Polygon -6459832 true true 67 122 96 126 63 144

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="false">
    <setup>setup-positions
setup-vacancies</setup>
    <go>go</go>
    <metric>ncl1</metric>
    <metric>ncl2</metric>
    <metric>ncl3</metric>
    <metric>ncl4</metric>
    <metric>ncl5</metric>
    <metric>ncl6</metric>
    <metric>ncl7</metric>
    <metric>ncl8</metric>
    <metric>ncl9</metric>
    <metric>ncl10</metric>
    <enumeratedValueSet variable="neighborhood">
      <value value="4"/>
      <value value="8"/>
      <value value="24"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-walls">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="strata-distribution">
      <value value="&quot;linear&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="n-of-strata">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="starting-stratum">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
