extensions [ profiler ]

globals [
  middle-edge    ; Two cell wide split between regions
  month year
  annual-rainfall-site-1  monthly-rainfall-site-1
  annual-rainfall-site-2  monthly-rainfall-site-2
  annual-rain-site1-drought annual-rain-site1-low annual-rain-site1-high



  enviro-function  social-function  landowned-function  ownland-function  conflict-function
  enviro-value-1   social-value-1   landowned-value-1   ownland-value-1   conflict-value-1
  enviro-value-2   social-value-2   landowned-value-2   ownland-value-2   conflict-value-2
  enviro-value-3   social-value-3   landowned-value-3   ownland-value-3   conflict-value-3

  enviro-list    social-list    landowned-list    ownland-list    conflict-list                       ; Lists from 0 to 100
  temp-list
  temp-score
  upper-bio   elev-mean  long-dist-moves-total        ;;REKHA EDITS                                                      ; Used in standardizing the environmental index

  comfort-threshold
  migration-tracker tick-max-mig tick-50-mig
]

patches-own [ elev soil-color soil-hue monthly-greenness  annual-biomass  current-biomass  biomass-per-cell  site-1? site-2?                                  ; Site 1 and 2 could be set with a single binary, but this will be clearer.
  is-land? owned? core-conflict? near-conflict? conflict-score temp? temp-count ]
breed [ people person ]
breed [ temp a-temp ]

people-own [ local-move-count distant-move-count social-network
  own-land       ; Own-land is now a score of 1, 2 or 3, for low, medium, or high
  area-1? area-2? origin dist_move        ;;;REKHA EDITS
  env-x  social-x  landowned-x  ownland-x  conflict-x
  env-y  social-y  landowned-y  ownland-y  conflict-y
  comfort-score  contentedness-index offtake temp-content conf-loc site-awareness

  current-score
  best-local-score  best-local-x  best-local-y
  best-distant-score  best-distant-x  best-distant-y
]

to setup
  clear-all
  setup-parms
  setup-landscape
  set annual-rain-site1-drought [400 420 380 350 310 305 280 350 380 405 410 425]
  set annual-rain-site1-low [197 231 208 199 198 247 230 250 224 228 249 224]
  set annual-rain-site1-high [646 585 574 537 624 635 634 604 580 578 626 618]
  get-annual-rainfall
  get-monthly-rainfall
  draw-production
  setup-people
  setup-parcels
  setup-conflict
  restore-defaults
  update-display
  test-to-hide-folks
  set migration-tracker []
  profiler:start
  reset-ticks
end



to go
  ; Each tick is one month ...
  test-to-hide-folks
  manage-time

  if month = 1 [ get-annual-rainfall ]                                ; Also clears current-biomass (and so biomass-per-cell by extension)
  get-monthly-rainfall
  draw-production

  ; ENVIRONMENT score needs standardized, but the target is changing each month.  I'm going to use mean + 2 * standard deviation as the upper limit for 100%
  ; That way, the score will be able to be near 100% regardless of the month
  let bio-mean mean [ current-biomass ] of patches with [ is-land? = TRUE ]
  let bio-sd standard-deviation [ current-biomass ] of patches with [ is-land? = TRUE ]
  set upper-bio  bio-mean + ( 2 * bio-sd )

  ask people [                        ; Everyone decides whether or not to move each month, and they can move back and forth as they wish (although frequent movements would suggest things are out of balance)
    set dist_move 0 ;; REKHA EDITS
    ; First, current location
    feed-livestock

    get-independent-vars xcor ycor    ; Get the x-axis values.  Some will need scaled.
    get-dependent-vars                ; Get the y-axis values.
    get-score
    set current-score temp-score

    ; Second, the local locations
    repeat 3 [                                                         ; REKHA EDITS Site-awarenes replaced with 3
      let x 0 let y 0 set best-local-score -999999
      ifelse area-1? = TRUE [
        ask one-of patches with [ site-1? = TRUE ] [                       ; NOTE sites match
          set x pxcor
          set y pycor
        ]
      ]
      [
        ask one-of patches with [ site-2? = TRUE ] [
          set x pxcor
          set y pycor
        ]
      ]
      get-independent-vars x y          ; Get the x-axis values.  Some will need scaled.
      get-dependent-vars                ; Get the y-axis values.
      get-score
      if temp-score > best-local-score [
        set best-local-score temp-score
        set best-local-x x
        set best-local-y y
      ]
    ]
    ; Third, the distant location
    repeat Site-awareness [
      let x 0 let y 0 set best-distant-score -999999
      ifelse area-1? = TRUE [
        ask one-of patches with [ site-2? = TRUE ] [                        ; NOTE sites don't match
          set x pxcor
          set y pycor
        ]
      ]
      [
        ask one-of patches with [ site-1? = TRUE ] [
          set x pxcor
          set y pycor
        ]
      ]
      get-independent-vars x y          ; Get the x-axis values.  Some will need scaled.
      get-dependent-vars                ; Get the y-axis values.
      get-score
      if temp-score > best-distant-score [
        set best-distant-score temp-score
        set best-distant-x x
        set best-distant-y y
      ]
    ]
    migrate-decision                  ; The main integration procedure.

  ]
  let temp-mig count people with [origin = 1 and distant-move-count > 0]
  set  migration-tracker lput temp-mig migration-tracker
  if year = 11 [stop]

  tick
  calc-summary
;  profiler:stop
;  print profiler:report
;  stop
end



to calc-summary
  let tot-mig  map [i -> i + 1] migration-tracker
  let tot-mig-enh lput (last tot-mig) tot-mig
  set tot-mig-enh remove-item 0 tot-mig-enh
  let temp-rate (map [[i j] ->  (j - i) / (last tot-mig)] tot-mig tot-mig-enh)
  let max-rate max temp-rate
  set tick-max-mig position max-rate temp-rate + 1
  let temp-mig-prop map [i ->  i / (last tot-mig)] tot-mig
  let temp-50 map [i -> (i - 0.6)] temp-mig-prop
  let low []
  foreach temp-50 [x -> ifelse x < 0 [set low lput 1 low] [set low lput x low]]
  set tick-50-mig position (min low) low

end





to feed-livestock
  ; Calculate offtake.  Assuming each cell is 250 m on a side.
  set offtake  Herd-size * ( ( ( 250 * 0.025 ) * 30 ) * 1000 )                       ; Herd x 250 kg x 2.5% per day x 30 days x kg to g
  let per-cell-offtake  offtake / 113                                                ; NOTE HARDWIRED VALUE THAT EQUALS IN-RADIUS 6.

  ; Going to try just removing biomass, and if the livestock don't get enough, that is ok, those animals will go hungry.
  ; IN-RADIUS 6 yields 113 patches.  That is a large area - we'll try that.
  ask patches in-radius 6 [
    set biomass-per-cell  biomass-per-cell - per-cell-offtake
    if biomass-per-cell < 0 [ set biomass-per-cell 0 ]
    set current-biomass  biomass-per-cell / ( 250 * 250 )
  ]
end


to get-score
  set comfort-score ( enviro-weight * env-y ) + ( social-weight * social-y ) + ( landowned-weight * landowned-y ) + ( ownland-weight * ownland-y ) + ( conflict-weight * conflict-y)
  set temp-score comfort-score
  if Debug? = TRUE [
    output-type who  output-type "   In area 1?  " output-type area-1?   output-type "   In area 2?  " output-type area-2?
    output-type "    Comfort Score:  " output-type precision comfort-score 1  output-type "   Contentedness:  "  output-type precision contentedness-index 1  output-type "                 Env-Y: " output-type precision env-y 2 output-type "  Social-Y: " output-type precision social-y 2 output-type "  Land Owned-Y: " output-type precision landowned-y 2 output-type "  Own Land-Y: " output-type precision ownland-y 2  output-type "  Conflict-Y: " output-print precision conflict-y 2
  ]
end

to migrate-decision
  ; SET THE MAXIMUM COMFORT SCORE to be able to calculate the deviation from the maximum score, to be incorporated into the contentedness score.
  ; set comfort-score     =      ( enviro-weight * env-y ) + ( social-weight * social-y ) + ( landowned-weight * landowned-y ) + ( ownland-weight * ownland-y ) + ( conflict-weight * conflict-y)
  let max-score ( enviro-weight * 100 ) + ( social-weight * 100 ) + ( landowned-weight * 100 ) + ( ownland-weight * 100 ) + ( conflict-weight * 100 )
  ; Return the comfort score and x and y values to the local (current) position.  These may change, but comfort-score is used and is only relevant locally, so it needs to be updated.
  get-independent-vars xcor ycor
  get-dependent-vars
  get-score
  if comfort-score <  comfort-threshold [
    set contentedness-index  contentedness-index - ( max-score - comfort-score )
    set temp-content contentedness-index
;    type "CONTENTED:   " type "          CONTENTEDNESS INDEX " type contentedness-index  type "Comfort threshold:  " type Comfort-threshold   type "           MAX SCORE " type max-score type "     COMFORT SCORE "  print comfort-score
  ]

  ; People in great spots don't move, but their contentedness indices can plumet FAR below zero (e.g., -10,000).
  ; There should be some chance that people will move when they are in nice locations but have contentedness indices less than 0.
  if contentedness-index <= 0 [
    ; MAY NOT MOVE if scores are not adequately high
    ifelse ( best-local-score > ( current-score + Short-migration-threshold ) ) or ( random-float 1.0 < 0.1 )
    [  ; Move locally, within the site in question
      setxy best-local-x best-local-y
      right random 360
      forward 0.5
      set contentedness-index ( Contentedness-reset-multiplier * max-score )
      set local-move-count local-move-count + 1
      set size size + 0.2
    ]
    [
      if ( best-distant-score > ( current-score + Long-migration-threshold ) ) ;   NO ... DON'T FORCE LONG-DISTANCE MIGRATION REGARDLESS OF CONTENTEDNESS-INDEX or ( random-float 1.0 < 0.05 )
      [
        setxy best-distant-x best-distant-y
        right random 360
        forward 0.5
        set area-1?  not area-1?
        set area-2?  not area-2?
        set contentedness-index ( Contentedness-reset-multiplier * max-score )
        set distant-move-count distant-move-count + 1
        set size size + 0.2
        set dist_move 1
      ]
    ]
  ]
  ; Update the X and Y values.  The values are updated as movements are considered, and so can be out of balance relative to the current location.  This isn't required, but will help align values stored with the local site.
;  type "CONTENTED:     CONFLICT  " type conflict-y type "          CONTENTEDNESS INDEX " type contentedness-index type "           MAX SCORE " type max-score type "     COMFORT SCORE "  print comfort-score
  get-independent-vars xcor ycor
  get-dependent-vars
  get-score
end


to get-independent-vars [ x y ]
  let ex 0  let l 0  let c 0

  ask patch x y [
    ; set ex monthly-greenness * 0.3921
    set ex ( current-biomass / upper-bio ) * 100.0
    if ex > 100 [ set ex 100 ]
    ifelse x < middle-edge
    [ set l Land-owned-site-1 ]
    [ set l Land-owned-site-2 ]
    set c conflict-score
  ]

  set env-x ex
  set social-x  social-network
  set landowned-x l
  if own-land = 1 [ set ownland-x 20 ]
  if own-land = 2 [ set ownland-x 50 ]
  if own-land = 3 [ set ownland-x 80 ]
  set conflict-x c
end

to get-dependent-vars
  set env-y item int ( env-x ) enviro-list
  set social-y item int ( social-x ) social-list
  set conflict-y item int ( conflict-x ) conflict-list
  set landowned-y item int ( landowned-x ) landowned-list
  set ownland-y item int ( ownland-x ) ownland-list

  if Debug? [
    type "Xs  Person: " type who type "   Env: " type precision env-x 2  type "   Social: " type precision social-x 2 type "   Landowned: " type landowned-x
    type "   Own land: " type ownland-x type "   Conflict: " print conflict-x
    type "Ys  Person: " type who type "   Env: " type precision env-y 2  type "   Social: " type precision social-y 2 type "   Landowned: " type landowned-y
    type "   Own land: " type ownland-y type "   Conflict: " print conflict-y
    print " "
  ]
end


to-report alint [ var x ]
  ; NOTE...NOTE: Assuming linear connections between individual points in the curves
  let x1 floor x
  let y1 item x1 var
  let x2 ceiling x
  let y2 item x2 var
  let x-remain x - x1                         ; Get remainder.  It will aways be a fractional measure given that x2 - x1 is always 1.
  let y y1 + ( ( y2 - y1 ) * x-remain )
  report y
end

to draw-production
  let per-meter  250 * 250
  ask patches with [ is-land? = TRUE ] [
    let c biomass-per-cell / per-meter
    set pcolor scale-color green c 0 600
  ]
end


to manage-time
  set month month + 1
  if month = 13 [
    set month 1
    set year year + 1
  ]
end

to get-annual-rainfall
  ; Making use of Deshmukh (2008) A common relationship between precipitation and grassland peak biomass for East and Southern Africa.  Af. Journal of Ecology 22:181-186.

  set annual-rainfall-site-1 random-normal Average-precipitation-Site-1 Precipitation-CV-Site-1
  ;set annual-rainfall-site-1 item year annual-rain-site1-drought ;; REKHA EDITS
  if annual-rainfall-site-1 < 100 [
    set annual-rainfall-site-1 100          ; Or set a higher minimum value? ... YES, setting a higher amount
  ]
  ifelse Use-the-same-climate-for-both-sites? = FALSE
  [
    set annual-rainfall-site-2 random-normal Average-precipitation-Site-2 Precipitation-CV-Site-2
    if annual-rainfall-site-2 < 100 [
      set annual-rainfall-site-2 100          ; Or set a higher minimum value?
    ]
  ]
  [
    set annual-rainfall-site-2 annual-rainfall-site-1
  ]

  ask patches with [ is-land? = TRUE ] [
    set current-biomass 0                                                               ; Biomass is not held over between years.
    ; MODIFYING ANNUAL BIOMASS to incorporate the spatial texture.  Using ELEV for that.  This will  include a hardwired stretch value
    if site-1? = TRUE [
      set annual-biomass ( 8.4888 * annual-rainfall-site-1 - 195.768 ) / 10.0             ; Divide 10 to convert to g / m2  from kg / ha
    ]
    if site-2? = TRUE [
      set annual-biomass ( 8.4888 * annual-rainfall-site-2 - 195.768 ) / 10.0             ; Divide 10 to convert to g / m2  from kg / ha
    ]
    let modifier  elev / elev-mean * 1.0
    set annual-biomass  annual-biomass * modifier
  ]
end


to get-monthly-rainfall
  ; Brute force to determine seasonality.  Coefficients should sum to 1.0
  if month = 1  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.05 ]
  if month = 2  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.06 ]
  if month = 3  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.10 ]
  if month = 4  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.18 ]
  if month = 5  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.15 ]
  if month = 6  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.08 ]
  if month = 7  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.01 ]
  if month = 8  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.02 ]
  if month = 9  [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.05 ]
  if month = 10 [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.10 ]
  if month = 11 [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.12 ]
  if month = 12 [ set monthly-rainfall-site-1 annual-rainfall-site-1 * 0.08 ]

  if month = 1  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.05 ]
  if month = 2  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.06 ]
  if month = 3  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.10 ]
  if month = 4  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.18 ]
  if month = 5  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.15 ]
  if month = 6  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.08 ]
  if month = 7  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.01 ]
  if month = 8  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.02 ]
  if month = 9  [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.05 ]
  if month = 10 [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.10 ]
  if month = 11 [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.12 ]
  if month = 12 [ set monthly-rainfall-site-2 annual-rainfall-site-2 * 0.08 ]

  ask patches with [ is-land? = TRUE ] [
    if month = 1  [ set current-biomass  current-biomass + ( annual-biomass * 0.05 ) ]
    if month = 2  [ set current-biomass  current-biomass + ( annual-biomass * 0.06 ) ]
    if month = 3  [ set current-biomass  current-biomass + ( annual-biomass * 0.10 ) ]
    if month = 4  [ set current-biomass  current-biomass + ( annual-biomass * 0.18 ) ]
    if month = 5  [ set current-biomass  current-biomass + ( annual-biomass * 0.15 ) ]
    if month = 6  [ set current-biomass  current-biomass + ( annual-biomass * 0.08 ) ]
    if month = 7  [ set current-biomass  current-biomass + ( annual-biomass * 0.01 ) ]
    if month = 8  [ set current-biomass  current-biomass + ( annual-biomass * 0.02 ) ]
    if month = 9  [ set current-biomass  current-biomass + ( annual-biomass * 0.05 ) ]
    if month = 10 [ set current-biomass  current-biomass + ( annual-biomass * 0.10 ) ]
    if month = 11 [ set current-biomass  current-biomass + ( annual-biomass * 0.12 ) ]
    if month = 12 [ set current-biomass  current-biomass + ( annual-biomass * 0.08 ) ]

    set biomass-per-cell current-biomass * 250 * 250                             ; Assuming cells 250 m on a side.  NOTE THE USE OF AN EXPLICIT SCALE
  ]
end


to update-display
  if Display-layer = "Soils" [
    paint-landscape
  ]
  if Display-layer = "Owned lands" [
    ask patches with [ owned? ] [
      set pcolor hsb ( soil-color * 0.50 + 20 ) 70 99
    ]
  ]
  if Display-layer = "Conflict zones" [
    ask patches with [ conflict-score > 0 ] [
      ; set pcolor hsb ( soil-color * 0.50 + 20 ) 70 ( 100 - conflict-score )
      set pcolor hsb 0 70 conflict-score
    ]
  ]
  if Display-layer = "Production" [
    ask patches with [ is-land? = TRUE ] [
      set pcolor scale-color green current-biomass 0 600
    ]
  ]
end


to test-to-hide-folks
  ifelse Hide-people?
  [ ask people [ set hidden? TRUE ] ]
  [ ask people [ set hidden? FALSE ] ]

  ask people [ set label "" ]
  if Show-local-moves? = TRUE
  [
    set Show-distant-moves? FALSE
    ask people [ set label local-move-count ]
  ]

  if Show-distant-moves? = TRUE
  [
    set Show-local-moves? FALSE
    ask people [ set label distant-move-count ]
  ]
end

to setup-people
  let S1 (Percent-of-people-in-Site-1 / 100) * 1000
  let max-score ( enviro-weight * 100 ) + ( social-weight * 100 ) + ( landowned-weight * 100 ) + ( ownland-weight * 100 ) + ( conflict-weight * 100 )
  set comfort-threshold ( max-score * Comfort-threshold-multiplier )
  create-people S1 [              ; Can be made a variable
    set shape "circle"
    set size 2
    set origin 1
    setxy random-xcor random-ycor
    while [ site-1? = FALSE ] [
      setxy random-xcor random-ycor
    ]
    set social-network random-normal social-network-mean social-network-sd
    if social-network < 0.0 [ set social-network 0.0 ]
    if social-network > 100.0 [ set social-network 100.0 ]
    set site-awareness  random-normal site-awareness-mean site-awareness-sd
    if site-awareness < 0.0 [ set site-awareness 1]
    set color red + ( random 4 - 2)
    set local-move-count 0
    set distant-move-count 0
    set own-land 0                        ; This is reset on a population basis below
    set area-1? TRUE
    set area-2? FALSE
    set contentedness-index ( Contentedness-reset-multiplier * max-score )
  ]
  create-people 1000 - S1 [              ; Can be made a variable
    set shape "circle"
    set size 2
    set origin 2
    setxy random-xcor random-ycor
    while [ site-2? = FALSE ] [
      setxy random-xcor random-ycor
    ]
    set social-network random-normal social-network-mean social-network-sd
    if social-network < 0.0 [ set social-network 0.0 ]
    if social-network > 100.0 [ set social-network 100.0 ]
    set color blue + ( random 4 - 2)
    set local-move-count 0
    set distant-move-count 0
    set own-land 0                         ; This is reset on a population basis below
    set area-1? FALSE
    set area-2? TRUE
    set contentedness-index ( Contentedness-reset-multiplier * max-score )
  ]
  let c int ( S1 * ( Percent-owning-land-site-1 / 100.0 ) )
  ask n-of c people with [ area-1? = TRUE ] [
    set own-land one-of [ 1 2 3 ]
  ]
  set c int ( (1000 - S1 ) * ( Percent-owning-land-site-2 / 100.0 ) )
  ask n-of c people with [ area-2? = TRUE ] [
    set own-land one-of [ 1 2 3 ]
  ]
end


to setup-conflict           ; Making this specific to location just for simplicity using a brute-force approach.
  if Core-area-in-conflict-site-1 > 0 [
    ask patches [
      set core-conflict? FALSE
      set temp? FALSE
      set conflict-score 0
    ]
    let core-cnt 0
    let core-target ( Core-area-in-conflict-site-1 / 100.0 ) * count patches with [ site-1? = TRUE ]
    create-temp 1 [
      setxy 290 100          ; START ON THE SITE-2 SO THAT TEMP? GETS SET BELOW.  Important.
      while [ site-1? = FALSE ] [
        setxy random-xcor random-ycor
        set temp? TRUE
        ask neighbors with [ site-1? = TRUE ] [ set temp? TRUE ]
      ]
    ]
    while [ core-cnt < core-target ] [                                      ; Note that this limit is approximate.  A few more cells may be owned than the math indicates
      ; Refresh the edges every 20 times edges expand
      ask patches with [ site-1? = TRUE and core-conflict? = TRUE ] [
        let temp-cnt count neighbors with [ core-conflict? = FALSE ]
        ifelse temp-cnt > 0
        [ set temp? TRUE ]
        [ set temp? FALSE ]
      ]

      repeat 20 [
        if any? patches with [ temp? = TRUE ] [
          ask one-of patches with [ temp? = TRUE ] [
            set temp? FALSE
            set pcolor red
            ask neighbors with [ core-conflict? = FALSE and site-1? = TRUE ] [                 ; Painting all neighbors, so may exceed limit, but not by too much.
              set core-conflict? TRUE
              set temp? TRUE
              set pcolor red
            ]
          ]
        ]
      ]
      set core-cnt count patches with [ core-conflict? = TRUE and site-1? = TRUE]
    ]
    ask patches with [ core-conflict? ] [ set conflict-score 100 ]
    ask temp [ die ]

    do-ring 100
    do-ring 90
    do-ring 80
    do-ring 70
    do-ring 60
    do-ring 50
    do-ring 40
    do-ring 30
    do-ring 20
    do-ring 10
  ]

  if Core-area-in-conflict-site-2 > 0 [
    let core-cnt 0
    let core-target ( Core-area-in-conflict-site-2 / 100.0 ) * count patches with [ site-2? = TRUE ]
    create-temp 1 [
      setxy 10 100          ; START ON THE SITE-1 SO THAT TEMP? GETS SET BELOW.  Important.
      while [ site-2? = FALSE ] [
        setxy random-xcor random-ycor
        set temp? TRUE
        ask neighbors with [ site-2? = TRUE ] [ set temp? TRUE ]
      ]
    ]
    while [ core-cnt < core-target ] [                                      ; Note that this limit is approximate.  A few more cells may be owned than the math indicates
      ; Refresh the edges every 20 times edges expand
      ask patches with [ site-2? = TRUE and core-conflict? = TRUE ] [
        let temp-cnt count neighbors with [ core-conflict? = FALSE ]
        ifelse temp-cnt > 0
        [ set temp? TRUE ]
        [ set temp? FALSE ]
      ]

      repeat 20 [
        if any? patches with [ temp? = TRUE ] [
          ask one-of patches with [ temp? = TRUE ] [
            set temp? FALSE
            set pcolor red
            ask neighbors with [ core-conflict? = FALSE and site-2? = TRUE ] [                 ; Painting all neighbors, so may exceed limit, but not by too much.
              set core-conflict? TRUE
              set temp? TRUE
              set pcolor red
            ]
          ]
        ]
      ]
      set core-cnt count patches with [ core-conflict? = TRUE and site-2? = TRUE]
    ]
    ask patches with [ core-conflict? ] [ set conflict-score 100 ]
    ask temp [ die ]

    do-ring 100
    do-ring 90
    do-ring 80
    do-ring 70
    do-ring 60
    do-ring 50
    do-ring 40
    do-ring 30
    do-ring 20
    do-ring 10
  ]
end


to do-ring  [ v ]
  let v2 v - 10
  ask patches with [ conflict-score = v and sum [ conflict-score ] of neighbors < ( v * 8 ) ] [
    ask neighbors with [ conflict-score < v and is-land? ] [
      set conflict-score v2
      set pcolor v - 5
    ]
  ]
  repeat 2 [
    ask patches with [ conflict-score = v2 and sum [ conflict-score ] of neighbors < ( v2 * 8 ) ] [
      ask neighbors with [ conflict-score = 0 and is-land? ] [
        set conflict-score v2
        set pcolor v - 5
      ]
    ]
  ]
end


to setup-landscape
  ; Set some initial conditions and values
  set middle-edge 150
  ask patches [
    set site-1? FALSE
    set site-2? FALSE
    ifelse pxcor >= middle-edge and pxcor <= middle-edge + 1
    [
      set pcolor white
      set is-land? FALSE
    ]
    [ set is-land? TRUE ]
    if pxcor < middle-edge [ set site-1? TRUE  set site-2? FALSE]
    if pxcor > ( middle-edge + 1 ) [ set site-1? FALSE  set site-2? TRUE]
    set owned? FALSE
    set temp? FALSE
  ]
  let n-bumps 0
  let meander 3
  let bumpiness 96

  ask patches with [ is-land? = TRUE ] [
    set elev 150
  ]

  ; Now add small texture to the landscape
  set n-bumps int ( sqrt ( world-width * world-height ) * ( bumpiness * 0.004 ) )
  ask n-of n-bumps patches with [ is-land? ] [
    let bump-size 1 + max-pxcor * 0.001 * ( random-float bumpiness )
    ask patches in-radius bump-size [
      set elev elev + ( bump-size - ( distance myself ) )
    ]
  ]

  ; Standarize elevation to be between 0 and 1, just for each of analyses
  let emin min [ elev ] of patches
  let emax max [ elev ] of patches
  ask patches with [ is-land? ] [
    set elev  ( elev - emin ) / ( emax - emin )
  ]

  ; Calculate soil-color, now that the DEM is complete.  Aim for drier yellow soils in the east, and darker richer soils in the west
  ask patches with [ is-land? ] [
    let xer  ( ( world-width -  pxcor ) + int ( world-width / 2 ) ) / world-width            ; Standardize the X position to a value between 0 and 1
    ; Add soil-color related to topography, but trimmed to 0 to 1
    set soil-color  ( ( xer * 1.5 ) + elev ) / 2.5
  ]

  ; Standarize AGAIN soil-color to be between 0 and 100, to make painting and modeling straightforward
  set emin min [ soil-color ] of patches
  set emax max [ soil-color ] of patches
  ask patches with [ is-land? ] [
    set soil-color 100 - ( int ( ( soil-color - emin ) / ( emax - emin ) * 100.0 ) )
  ]

  paint-landscape

  ; Save the RGB of the patches.  NOTE that the HSB complexity is not relevant once colors are assigned to soil.
  ask patches with [ is-land? ] [
    let temp-hsb extract-hsb pcolor
    set soil-hue item 0 temp-hsb
  ]

  set elev-mean mean [ elev ] of patches with [ is-land? = TRUE ]

end


to paint-landscape
  ask patches with [ is-land? ] [
    set pcolor hsb ( soil-color * 0.50 + 20 ) 70 70
  ]
end


to setup-parcels
  ; Do calculations to set limits to grow to and numbers of owners to grow.  NOTE that people can live on owned land and not be designated initial owners.
  let num-owning-land  ( count people with [ area-1? = TRUE ] ) * ( Percent-owning-land-site-1 / 100.0 )
  ask n-of num-owning-land people with [ area-1? = TRUE ] [
    set owned? TRUE                      ; Patch
  ]
  ; Initialize the patches occupied by owners as yellow and owned.
  ask patches with [ owned? = TRUE and site-1? = TRUE ] [
    set pcolor yellow
    ask neighbors with [ site-1? = TRUE ] [ set temp? TRUE ]
  ]
  let total-patches-owned  count patches with [ site-1? = TRUE ] * ( Land-owned-site-1 / 100.0 )    ; The limit to which to grown
  let own 0

  while [ own < total-patches-owned ] [                                      ; Note that this limit is approximate.  A few more cells may be owned than the math indicates
    ; Refresh the edges every 50 times edges expand
    ask patches with [ site-1? = TRUE and owned? = TRUE ] [
      set temp-count count neighbors with [ owned? = FALSE and site-1? = TRUE ]
      ifelse temp-count > 0
      [ set temp? TRUE ]
      [ set temp? FALSE ]
    ]

    repeat 50 [
      ask one-of patches with [ temp? = TRUE ] [
        set temp? FALSE
        set pcolor yellow
        ask neighbors with [ owned? = FALSE and site-1? = TRUE ] [                 ; Painting all neighbors, so may exceed limit, but not by too much.
          set owned? TRUE
          set temp? TRUE
          set pcolor yellow
        ]
      ]
    ]
    set own count patches with [ site-1? = TRUE and owned? = TRUE ]
  ]

  ;********************************************************************************************************
  ; Do calculations to set limits to grow to and numbers of owners to grow.  NOTE that people can live on owned land and not be designated initial owners.
  set num-owning-land  ( count people with [ area-2? = TRUE ] ) * ( Percent-owning-land-site-2 / 100.0 )
  ask n-of num-owning-land people with [ area-2? = TRUE ] [
    set owned? TRUE                      ; Patch
  ]
  ; Initialize the patches occupied by owners as yellow and owned.
  ask patches with [ owned? = TRUE and site-2? = TRUE ] [
    set pcolor yellow
    ask neighbors with [ site-2? = TRUE ] [ set temp? TRUE ]
  ]
  set total-patches-owned  count patches with [ site-2? = TRUE ] * ( Land-owned-site-2 / 100.0 )    ; The limit to which to grown
  set own 0

  while [ own < total-patches-owned ] [                                      ; Note that this limit is approximate.  A few more cells may be owned than the math indicates
    ; Refresh the edges every 50 times edges expand
    ask patches with [ site-2? = TRUE and owned? = TRUE ] [
      set temp-count count neighbors with [ owned? = FALSE and site-2? = TRUE ]
      ifelse temp-count > 0
      [ set temp? TRUE ]
      [ set temp? FALSE ]
    ]

    repeat 50 [
      ask one-of patches with [ temp? = TRUE ] [
        set temp? FALSE
        set pcolor yellow
        ask neighbors with [ owned? = FALSE and site-2? = TRUE ] [                 ; Painting all neighbors, so may exceed limit, but not by too much.
          set owned? TRUE
          set temp? TRUE
          set pcolor yellow
        ]
      ]
    ]
    set own count patches with [ site-2? = TRUE and owned? = TRUE ]
  ]

  ; Paint the final image of owned lands
  paint-landscape
  ask patches with [ owned? ] [
    set pcolor hsb ( soil-color * 0.50 + 20 ) 70 99
  ]

end












to setup-parms
  set enviro-list []
  set social-list []
  set landowned-list []
  set ownland-list []
  set conflict-list []

  set temp-list []
end


to restore-defaults
  set-current-plot "Coefficients-1"
  clear-plot
  set-current-plot-pen "Environment"
  set enviro-function  "Sigmoid"   set enviro-value-1 0  set enviro-value-2 0  set enviro-value-3 0           ; Values aer not used
  plot-coefficient enviro-function enviro-value-1 enviro-value-2 enviro-value-3
  set enviro-list temp-list
  set-current-plot-pen "Conflict"
  set conflict-function  "Asymptotic"   set conflict-value-1 100.0  set conflict-value-2 0.04  set conflict-value-3 0             ; Note asymptote is higher than end result.
  plot-coefficient conflict-function conflict-value-1 conflict-value-2 conflict-value-3
  set conflict-list temp-list
  set-current-plot-pen "LandOwned"
  set landowned-function  "Linear"   set landowned-value-1 100  set landowned-value-2 -1.0  set landowned-value-3 0
  plot-coefficient landowned-function landowned-value-1 landowned-value-2 landowned-value-3
  set landowned-list temp-list
  set-current-plot "Coefficients-2"
  clear-plot
  set-current-plot-pen "Social"
  set social-function  "Sigmoid"   set social-value-1 0  set social-value-2 0  set social-value-3 0           ; Values aer not used
  plot-coefficient social-function social-value-1 social-value-2 social-value-3
  set social-list temp-list
  set-current-plot-pen "OwnLand"
  let x 0
  set temp-list []
  while [ x < 33 ] [ set temp-list lput Low-owned-land temp-list  set x x + 1 ]
  while [ x < 67 ] [ set temp-list lput Mid-owned-land temp-list  set x x + 1 ]
  while [ x < 100 ] [ set temp-list lput High-owned-land temp-list  set x x + 1 ]
  set ownland-list temp-list
  set x 0
  while [ x < 100 ] [
    let y item x ownland-list
    plotxy  x  y
    set x x + 1
  ]

  set Focal-relationship "Environment"
  set Function-family enviro-function
  set Value-1 enviro-value-1
  set Value-2 enviro-value-2
  set Value-3 enviro-value-3
end


to plot-coefficient   [ c v1 v2 v3 ]
  let x -5.0
  let y 0
  set temp-list []
  if c = "Sigmoid" [
    while [ x < 5.0 ] [
      set y ( 1 / ( 1 + exp ( -1 * x ) ) * 100 )
      let tx int ( x * 10 + 50 )
      plotxy tx y
      set temp-list lput y temp-list
      set x x + 0.1
    ]
  ]                                                      ; Confirmed 101 entries in temp-list for SIGMOID, as needed, 0 to 100.
  set x 1
  let d 1 ; v2 - v3                                          ; Shift the curve to the left or right to match the X intercept requested.  Also need to add it to the X loop
  if c = "Asymptotic" [
    while [ x <= ( 100 + d ) ] [
    ;  set y -1 * ( v1 * ( v2 - x ) ) / x                ; V2 shapes the line, x3 is x intercept, x1 is asymptote
    ; REPLACING THIS WITH A DECAYING FUNCTION
      set y v1 * ( 1 - v2 ) ^ x
      let tx x - d
      if y >= 0  [
        set temp-list lput y temp-list
        plotxy  tx  y
      ]
      set x x + 1
    ]
  ]                                                       ; Confirmed 101 entries in temp-list for Asymptotic, as needed
  set x 0
  set y 0
  if c = "Linear" [
    while [ x <= 100.0 ] [
      set y v1 + v2 * x
      set temp-list lput y temp-list
      plotxy x y
      set x x + 1
    ]
  ]                                                        ; Confirmed 101 entries in temp-list for Asymptotic, as needed
  set x -10
  set y 0
  if c = "Quadratic" [
    while [ x < 10.0 ] [
      set y ( v1 * x * x ) + v2 * x + v3
;      type x type "  " print y
      let tx ( x + 10 ) * 5
      set temp-list lput y temp-list
      plotxy tx y
      set x x + 0.2
    ]
  ]                                                         ; Confirmed 101 entries in temp-list for Asymptotic, as needed
end

to update-function
  if Focal-relationship = "Environment" [
    set-current-plot "Coefficients-1"
    clear-plot
    set-current-plot-pen "Environment"
    set enviro-function  Function-family   set enviro-value-1 Value-1  set enviro-value-2 Value-2  set enviro-value-3 Value-3
    plot-coefficient enviro-function enviro-value-1 enviro-value-2 enviro-value-3
    set enviro-list temp-list
    ; Plot the remaining entries.  No need to recalculate, use existing lists
    set-current-plot-pen "Conflict"     one-plot conflict-list
    set-current-plot-pen "Availland"    one-plot landowned-list
  ]
  if Focal-relationship = "Conflict" [
    set-current-plot "Coefficients-1"
    clear-plot
    set-current-plot-pen "Conflict"
    set conflict-function  Function-family   set conflict-value-1 Value-1  set conflict-value-2 Value-2  set conflict-value-3 Value-3
    plot-coefficient conflict-function conflict-value-1 conflict-value-2 conflict-value-3
    set conflict-list temp-list
    ; Plot the remaining entries.  No need to recalculate, use existing lists
    set-current-plot-pen "Environment"     one-plot enviro-list
    set-current-plot-pen "Availland"    one-plot landowned-list
  ]
  if Focal-relationship = "Available Land" [
    set-current-plot "Coefficients-1"
    clear-plot
    set-current-plot-pen "Available land"
    set landowned-function  Function-family   set landowned-value-1 Value-1  set landowned-value-2 Value-2  set landowned-value-3 Value-3
    plot-coefficient landowned-function landowned-value-1 landowned-value-2 landowned-value-3
    set landowned-list temp-list
    ; Plot the remaining entries.  No need to recalculate, use existing lists
    set-current-plot-pen "Conflict"     one-plot conflict-list
    set-current-plot-pen "Environment"    one-plot enviro-list
  ]

  if Focal-relationship = "Environment" [
    set-current-plot "Coefficients-2"
    clear-plot
    set-current-plot-pen "Social"
    set enviro-function  Function-family   set enviro-value-1 Value-1  set enviro-value-2 Value-2  set enviro-value-3 Value-3           ; Values aer not used
    plot-coefficient enviro-function enviro-value-1 enviro-value-2 enviro-value-3
    set enviro-list temp-list
    ; Plot the remaining entries.  No need to recalculate, use existing lists
    set-current-plot-pen "Ownland"     one-plot conflict-list
  ]
  if Focal-relationship = "Owning land" [
    set-current-plot "Coefficients-2"
    clear-plot
    set-current-plot-pen "Ownland"
    set ownland-function  Function-family   set ownland-value-1 Value-1  set ownland-value-2 Value-2  set ownland-value-3 Value-3           ; Values aer not used
    plot-coefficient ownland-function ownland-value-1 ownland-value-2 ownland-value-3
    set ownland-list temp-list
    ; Plot the remaining entries.  No need to recalculate, use existing lists
    set-current-plot-pen "Social"     one-plot social-list
  ]
end

to one-plot [ lst ]
  let i 0
  while [ i <= 100 ] [
    let y item i lst
    plotxy i y
    set i i + 1
  ]
end


to show-values
  if Focal-relationship = "Environment"    [ set Function-family enviro-function     set Value-1 enviro-value-1   set Value-2 enviro-value-2   set Value-3 enviro-value-3 ]
  if Focal-relationship = "Social network" [ set Function-family social-function     set Value-1 social-value-1   set Value-2 social-value-2   set Value-3 social-value-3 ]
  if Focal-relationship = "Conflict"       [ set Function-family conflict-function   set Value-1 conflict-value-1   set Value-2 conflict-value-2   set Value-3 conflict-value-3 ]
  if Focal-relationship = "Available land" [ set Function-family landowned-function  set Value-1 landowned-value-1   set Value-2 landowned-value-2   set Value-3 landowned-value-3 ]
  if Focal-relationship = "Owning land"    [ set Function-family ownland-function    set Value-1 ownland-value-1   set Value-2 ownland-value-2   set Value-3 ownland-value-3 ]
end


to force-2-to-1
  set Average-precipitation-site-2 Average-precipitation-site-1
  set Precipitation-CV-site-2 Precipitation-CV-site-1
  set Land-owned-site-2 Land-owned-site-1
  set Percent-owning-land-site-2 Percent-owning-land-site-1
  set Core-area-in-conflict-site-2 Core-area-in-conflict-site-1
end




to restore-slider-defaults
  set Precipitation-CV-site-2  30
  set Long-migration-threshold  17
  set Percent-owning-land-site-2  20
  set Show-local-moves?  false
  set Social-network-mean  24
  set Display-layer  "Conflict zones"
  set Show-distant-moves?  false
  set LandOwned-weight  1
  set High-owned-land  10
  set Percent-of-people-in-Site-1  50
  set Debug?  false
  set Average-precipitation-site-1  230
  set Land-owned-site-1  10
  set Average-precipitation-site-2  400
  set Social-weight  1
  set Focal-relationship  "Environment"
  set Herd-size  20
  set Comfort-threshold-multiplier  0.39
  set Conflict-weight  1
  set Enviro-weight  1
  set Social-network-sd  22
  set Low-owned-land  10
  set Short-migration-threshold  10
  set Ownland-weight  1
  set Core-area-in-conflict-site-1  3
  set Contentedness-reset-multiplier  3.0
  set Core-area-in-conflict-site-2  3
  set Function-family  "Sigmoid"
  set Mid-owned-land  90
  set Site-awareness-mean  3
  set Site-awareness-sd 0
  set Value-1  0
  set Hide-people?  false
  set Value-2  0
  set Use-the-same-climate-for-both-sites?  false
  set Value-3  0
  set Precipitation-CV-site-1  30
  set Percent-owning-land-site-1  20
  set Land-owned-site-2  25

end
@#$#@#$#@
GRAPHICS-WINDOW
232
64
958
549
-1
-1
2.38
1
10
1
1
1
0
0
0
1
0
301
0
199
0
0
1
ticks
30.0

BUTTON
12
10
76
43
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
88
10
151
43
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
325
1109
452
1142
Hide-people?
Hide-people?
1
1
-1000

SLIDER
13
163
220
196
Land-owned-site-1
Land-owned-site-1
0
100
10.0
1
1
percent
HORIZONTAL

SLIDER
13
207
221
240
Percent-owning-land-site-1
Percent-owning-land-site-1
0
100
20.0
1
1
percent
HORIZONTAL

CHOOSER
604
12
742
57
Display-layer
Display-layer
"Production" "Soils" "Owned lands" "Conflict zones"
3

BUTTON
747
19
819
52
Update
update-display
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
13
85
218
118
Average-precipitation-site-1
Average-precipitation-site-1
150
600
240.0
10
1
mm
HORIZONTAL

SLIDER
13
118
217
151
Precipitation-CV-site-1
Precipitation-CV-site-1
0
60
30.0
1
1
NIL
HORIZONTAL

PLOT
10
665
332
868
Precipitation
Month
Precipitation
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Annual-1" 1.0 0 -13345367 true "" "plot annual-rainfall-site-1"
"pen-1" 1.0 0 -7500403 true "" "plot annual-rainfall-site-2"

SLIDER
13
319
220
352
Social-network-mean
Social-network-mean
0
100
24.0
1
1
NIL
HORIZONTAL

SLIDER
13
353
220
386
Social-network-sd
Social-network-sd
0
30
22.0
1
1
NIL
HORIZONTAL

SLIDER
13
252
221
285
Core-area-in-conflict-site-1
Core-area-in-conflict-site-1
0
30
3.0
1
1
percent
HORIZONTAL

PLOT
638
950
948
1150
Coefficients-1
Response strength
Coefficient
0.0
100.0
0.0
100.0
true
true
"" ""
PENS
"Environment" 1.0 0 -14439633 true "" ""
"LandOwned" 1.0 0 -5825686 true "" ""
"Conflict" 1.0 0 -2674135 true "" ""

PLOT
958
949
1257
1149
Coefficients-2
Response strength
Coefficient
0.0
100.0
0.0
100.0
false
true
"" ""
PENS
"Social" 1.0 0 -16777216 true "" ""
"OwnLand" 1.0 0 -14070903 true "" ""

SWITCH
470
1109
597
1142
Debug?
Debug?
1
1
-1000

PLOT
341
667
640
870
Mean Contentedness Index
Month
Contentedness
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Site-1" 1.0 0 -16777216 true "" "plot mean [ contentedness-index ] of people with [ area-1? = TRUE ]"
"Site-2" 1.0 0 -5298144 true "" "plot mean [ contentedness-index ] of people with [ area-2? = TRUE ]"

MONITOR
17
614
74
659
Month
month
0
1
11

MONITOR
83
614
140
659
Year
year
0
1
11

CHOOSER
16
917
175
962
Focal-relationship
Focal-relationship
"Environment" "Social network" "Available land" "Owning land" "Conflict"
0

CHOOSER
17
971
176
1016
Function-family
Function-family
"Linear" "Sigmoid" "Asymptotic" "Quadratic"
1

INPUTBOX
194
918
288
978
Value-1
0.0
1
0
Number

INPUTBOX
194
985
288
1045
Value-2
0.0
1
0
Number

INPUTBOX
194
1052
288
1112
Value-3
0.0
1
0
Number

TEXTBOX
31
1157
203
1227
Linear - y=Intercept + Slope*x\n  Value-1 = Intercept\n  Value-2 = Slope\n  Value-3 = (Not used)
11
0.0
1

TEXTBOX
199
1158
349
1214
Sigmoid - 1 / (1 + e^-x)\n  Value-1 = (Not used)\n  Value-2 = (Not used)\n  Value-3 = (Not used)
11
0.0
1

BUTTON
105
1027
177
1060
Update
update-function
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
23
1105
177
1138
Default functions
restore-defaults
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
341
1157
491
1213
Asymptotic - rX * (1-X/K)\n  Value-1 = r\n  Value-2 = K\n  Value-3 = (Not used)
11
0.0
1

TEXTBOX
21
1069
171
1087
Note: Plots may overlap.
11
0.0
1

BUTTON
20
1026
91
1059
Display
show-values
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
488
1157
638
1213
Quadradic - y=ax^2+bx+c\n  Value-1 = a\n  Value-2 = b\n  Value-3 = c
11
0.0
1

SLIDER
360
915
532
948
Enviro-weight
Enviro-weight
0
1.0
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
361
950
533
983
Social-weight
Social-weight
0
1.0
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
361
985
533
1018
Conflict-weight
Conflict-weight
0
1.0
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
362
1020
534
1053
LandOwned-weight
LandOwned-weight
0
1.0
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
363
1056
535
1089
Ownland-weight
Ownland-weight
0
1.0
1.0
0.01
1
NIL
HORIZONTAL

SLIDER
970
83
1175
116
Average-precipitation-site-2
Average-precipitation-site-2
150
600
400.0
10
1
mm
HORIZONTAL

SLIDER
970
115
1176
148
Precipitation-CV-site-2
Precipitation-CV-site-2
0
60
30.0
1
1
NIL
HORIZONTAL

SWITCH
238
10
500
43
Use-the-same-climate-for-both-sites?
Use-the-same-climate-for-both-sites?
1
1
-1000

SLIDER
971
160
1179
193
Land-owned-site-2
Land-owned-site-2
0
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
972
205
1180
238
Percent-owning-land-site-2
Percent-owning-land-site-2
0
100
24.0
1
1
NIL
HORIZONTAL

SLIDER
973
249
1179
282
Core-area-in-conflict-site-2
Core-area-in-conflict-site-2
0
30
3.0
1
1
NIL
HORIZONTAL

BUTTON
987
302
1165
335
Force site 2 to equal site 1
force-2-to-1
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
22
66
172
84
Sending site parameters
11
0.0
1

TEXTBOX
23
297
173
315
Individual parameters
11
0.0
1

TEXTBOX
980
63
1130
81
Receiving site parameters
11
0.0
1

SLIDER
16
522
224
555
Long-migration-threshold
Long-migration-threshold
0
100
17.0
1
1
index
HORIZONTAL

SLIDER
14
448
222
481
Site-awareness-mean
Site-awareness-mean
0
100
3.0
1
1
sites
HORIZONTAL

SLIDER
14
560
223
593
Short-migration-threshold
Short-migration-threshold
0
100
10.0
1
1
index
HORIZONTAL

MONITOR
179
613
307
658
Average local moves
mean [ local-move-count ] of people
3
1
11

MONITOR
296
613
428
658
Maximum local moves
max [ local-move-count ] of people
0
1
11

TEXTBOX
699
1164
1142
1206
Note: Available land function flipped vertically.\n      Conflict function flipped vertically.
11
0.0
1

TEXTBOX
822
1205
1126
1245
Low scores may prompt movement.
16
0.0
1

TEXTBOX
400
558
447
576
Site 1
14
0.0
1

TEXTBOX
761
561
911
579
Site 2
14
0.0
1

SLIDER
972
454
1229
487
Contentedness-reset-multiplier
Contentedness-reset-multiplier
0
10.0
3.0
0.1
1
NIL
HORIZONTAL

SLIDER
972
415
1229
448
Herd-size
Herd-size
0
100
20.0
1
1
cattle
HORIZONTAL

TEXTBOX
981
394
1131
412
Global parameters
11
0.0
1

TEXTBOX
240
45
528
73
Note: Site-2 sliders for climate not used if on.
11
0.0
1

SLIDER
973
506
1232
539
Percent-of-people-in-Site-1
Percent-of-people-in-Site-1
0
100
50.0
1
1
percent
HORIZONTAL

TEXTBOX
981
543
1144
572
Note: Of 1000 people
11
0.0
1

MONITOR
450
613
592
658
Average distant moves
mean [ distant-move-count ] of people
3
1
11

MONITOR
593
612
738
657
Maximum distant moves
max [ distant-move-count ] of people
3
1
11

SWITCH
878
21
1046
54
Show-local-moves?
Show-local-moves?
1
1
-1000

SWITCH
1064
20
1246
53
Show-distant-moves?
Show-distant-moves?
1
1
-1000

BUTTON
1044
587
1168
620
Restore defaults
restore-slider-defaults
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
1024
663
1057
872
Low-owned-land
Low-owned-land
0
100
10.0
1
1
coeff.
VERTICAL

SLIDER
1089
662
1122
870
Mid-owned-land
Mid-owned-land
0
100
90.0
1
1
coeff.
VERTICAL

SLIDER
1158
662
1191
866
High-owned-land
High-owned-land
0
100
10.0
1
1
coeff.
VERTICAL

TEXTBOX
1074
881
1224
899
Owned-land settings
11
0.0
1

SLIDER
14
396
223
429
Comfort-threshold-multiplier
Comfort-threshold-multiplier
0
1.0
0.39
0.01
1
NIL
HORIZONTAL

SLIDER
14
483
220
516
site-awareness-sd
site-awareness-sd
0
5
0.0
1
1
NIL
HORIZONTAL

PLOT
651
666
917
870
Migrations to receiving site
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count turtles with [origin = 1 and distant-move-count > 0]"

@#$#@#$#@
## WHAT IS IT?

MLSM is a theoretical model that integrates concepts from migration and land system sciences. Its purpose is to support the development and testing of theories that articulate the role of rural migration and land system interactions in driving coupled human-environment systems dynamics in frontier landscapes, and in forecasting their future behaviors. MSLM represents a stylized agropastoral system with two landscapes; a sending landscape from which agents may emigrate and a frontier-style receiving landscape into which agents can immigrate.    

## HOW IT WORKS

Agents are agropastoral households with livestock herds, a local social network, owned land, and a fixed number of local and non-local sites that they have awareness of. 

At Setup, agents are distributed on patches with randomly assigned levels of conflict and land ownsership. During the simualtion, spatio-temporally varying precipitation  drives biomass production on each patch. 

Agent and patch-specific factors determine the level of comfort each agent experiences while being located on a patch. When this comfort level is lower than a threshold value, agents experience a decline in their contentedness score.

The contentedness score ranges between 0 and a maximum value and represents the cumulative satisfaction an agent experiences when located on a particular patch.

When an agent's contentedness score reaches 0, they begin to assess the suitability of other local and non-local patches that they are aware of. 

Agents may move to a suitable patch, if one is found, resetting their contentedness score to the maxmum value. When an alternative patch is not avaialble, agents remain on the patch.

## HOW TO USE IT

### Setup: Sets up the model

### Go: Runs the model

### Setup patches in the receiving and sending sites

Average-precipitaton-site-1(2): Sets mean annual precipitation in the sending site (receiving site)

Precipitation-CV-site-1(2): Sets the coeffecient of variation around mean annual precipitation in the sending (receiving) site.

Land-owned-site-1(2): The fraction of land in the sending (receiving) site that is owned

Percent-owling-land-site-1(2): The fraction of agents in the sending (receiving) site that own land

Core-area-in-conflict-site-1(2): The fraction of patches in sending (receiving) site that experiences conflict


### Set up agents

Site-awareness-mean: The mean number of local and non-local sites that agents are aware of

Site-awareness-sd: The standard deviation associated with the site-awareness-mean


Set up global variables

Percent-of-people-in-Site-1: The fraction of agents that are initiated in the sending site

Herd-size: The number of livestock owned by each agent.


## THINGS TO TRY

1.  At first, run the model by setting up the receiving site as a frontier landscape with low population density, low conflicts and land ownsership.

2.  Holding all other variables constant, vary precipitation levels in the sending site and observe changes in migration rates

3.  Repeat above exercise, by similarly varying other variables associated with the sending site and agents.

## HOW TO CITE
Warrier R, Boone R, Salerno J. 2023. Migration land systems model: a theoretical agent-based model. Mountain Scholar. DOI: 10.25675/10217/236080
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="5" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [ contentedness-index ] of people with [ has-moved? = FALSE ]</metric>
    <metric>count people with [ has-moved? = FALSE ]</metric>
    <steppedValueSet variable="Land-owned" first="10" step="10" last="90"/>
    <steppedValueSet variable="Percent-owning-land" first="10" step="10" last="90"/>
  </experiment>
  <experiment name="experiment" repetitions="6" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>sum [offtake] of people with [area-2? = TRUE]</metric>
    <enumeratedValueSet variable="Site-awareness-mean">
      <value value="3"/>
      <value value="10"/>
      <value value="20"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="structure-agency" repetitions="1" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count people with [distant-move-count &gt; 0 and origin = 1]</metric>
    <metric>tick-max-mig</metric>
    <metric>tick-50-mig</metric>
    <steppedValueSet variable="Social-network-mean" first="0" step="20" last="100"/>
    <steppedValueSet variable="Core-area-in-conflict-site-1" first="1" step="5" last="30"/>
    <steppedValueSet variable="Site-awareness-mean" first="1" step="3" last="20"/>
    <steppedValueSet variable="Average-precipitation-site-1" first="200" step="50" last="600"/>
    <enumeratedValueSet variable="Percent-of-people-in-site-1">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Land-owned-site-1">
      <value value="85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Percent-owning-land-site-1">
      <value value="70"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social-network-sd">
      <value value="22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Comfort-threshold-multiplier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Average-precipitation-site-2">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Precipitation-CV-site-2">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Land-owned-site-2">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Percent-owning-land-site-2">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Core-area-in-conflict-site-2">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Herd-size">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Long-migration-threshold">
      <value value="17"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Short-migration-threshold">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>sum [offtake] of people with [area-2? = TRUE]</metric>
    <enumeratedValueSet variable="Precipitation-CV-site-2">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Long-migration-threshold">
      <value value="17"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Percent-owning-land-site-2">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-local-moves?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social-network-mean">
      <value value="24"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Display-layer">
      <value value="&quot;Conflict zones&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-distant-moves?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="LandOwned-weight">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="High-owned-land">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Site-awareness-mean">
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Percent-of-people-in-Site-1">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Debug?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Average-precipitation-site-1">
      <value value="600"/>
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Land-owned-site-1">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="site-awareness-sd">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Average-precipitation-site-2">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social-weight">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Focal-relationship">
      <value value="&quot;Environment&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Herd-size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Conflict-weight">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Enviro-weight">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Contentedness-reset-multiplier">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social-network-sd">
      <value value="22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Low-owned-land">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Short-migration-threshold">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ownland-weight">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Core-area-in-conflict-site-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Comfort-threshold-multiplier">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Core-area-in-conflict-site-2">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Function-family">
      <value value="&quot;Sigmoid&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Mid-owned-land">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hide-people?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Value-1">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Value-2">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use-the-same-climate-for-both-sites?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Value-3">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Land-owned-site-2">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Percent-owning-land-site-1">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Precipitation-CV-site-1">
      <value value="30"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>sum [offtake] of turtles with [origin = 1 and distant-move-count &gt; 0]</metric>
    <metric>annual-rainfall-site-1</metric>
    <metric>mean [site-awareness] of people</metric>
    <enumeratedValueSet variable="Site-awareness-mean">
      <value value="10"/>
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
